// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Collections.Generic;
using System.Net;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Communication
{
    /// <summary>
    /// f[^M@\񋟂 <see cref="IHttpSender{TPram}"/> ̊NXłB
    /// </summary>
    /// <remarks>
    /// <para>
    /// ̃NX͒ۃNXłB{Iȏt[ƒ萔ANGXg̏WbN񋟂܂B
    /// hNX쐬ꍇA<typeparamref name="TParam"/>őMf[^̌^w肵܂B
    /// </para>
    /// </remarks>
    /// <typeparam name="TParam">
    /// Mɓnp[^̌^B
    /// </typeparam>
    public abstract class SenderBase<TParam> : IHttpSender<TParam> where TParam : class
    {
        /// <summary>
        /// <see cref="ILog"/> NX̃CX^XłB
        /// </summary>
        /// <remarks>
        /// Oo͂ɗp܂B
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(SenderBase<TParam>));

        /// <summary>
        /// ɑMf[^(byte)̃ftHglłB
        /// </summary>
        /// <remarks>萔̒l 8192 łB</remarks>
        protected const int BUFFER_SIZE = 1024 * 8;

        /// <summary>
        /// HTTP NGXg POST \bhw肷ۂɗp镶łB
        /// </summary>
        /// <remarks>萔̒l "POST" łB</remarks>
        protected static readonly string POST_METHOD_NAME = "POST";

        /// <summary>
        /// HTTP wb_ Content-Disposition wb_ݒ肷ہAL[Ƃėp镶łB
        /// </summary>
        /// <remarks>
        /// <para>萔̒l "Content-Disposition" łB</para>
        /// </remarks>
        protected static readonly string CONTENT_DISPOSITION = "Content-Disposition";

        /// <summary>
        /// ɑMf[^(byte)łB
        /// </summary>
        private int _bufferSize = BUFFER_SIZE;

        /// <summary>
        /// HTTP NGXgɎw肷 Content-Type wb_̒l擾܂B
        /// </summary>
        public abstract string ContentType
        {
            get;
        }

        /// <summary>
        /// HTTP NGXgɎw肷 Method wb_̒l擾܂B
        /// </summary>
        /// <remarks>
        /// ̃vpeB͒ʏ "POST" Ԃ܂B
        /// </remarks>
        public virtual string Method
        {
            get
            {
                return POST_METHOD_NAME;
            }
        }

        /// <summary>
        /// ɑMf[^( byte )\l擾܂͐ݒ肵܂B
        /// </summary>
        /// <remarks>
        /// 0 ȉ̒lݒ肵ꍇA <see cref="SenderBase{TParam}.BUFFER_SIZE"/> ̒lp܂B
        /// </remarks>
        public int BufferSize
        {
            get
            {
                return _bufferSize;
            }
            set
            {
                if (value <= 0)
                {
                    _bufferSize = BUFFER_SIZE;
                }
                else
                {
                    _bufferSize = value;
                }
            }
        }

        /// <summary>
        /// <paramref name="request"/> œnꂽ <see cref="HttpWebRequest"/> pāA
        /// <paramref name="paramData"/>  <paramref name="headerCollection"/> ̏
        /// M܂B
        /// </summary>
        /// <remarks>
        /// <para>
        /// ̃\bh <see cref="SenderBase{TParam}.PrepareRequest"/> păNGXgIuWFNg܂B
        /// ̌A <see cref="SenderBase{TParam}.SendRequest"/> ĂяođMs܂B
        /// </para>
        /// <para>
        /// ʏÃ\bh𒼐ڃI[o[Ch邱Ƃ͔ĂB <see cref="Send"/> \bh
        /// <see cref="PrepareRequest"/> \bhŃNGXgA<see cref="SendRequest"/> 
        /// \bhւƏϏ܂B̓ɖ肪Ȃ΁A <see cref="SendRequest"/> \bhŏ
        /// ̖{̂Lq܂B
        /// </para>
        /// </remarks>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="request"/> , <paramref name="paramData"/> , <paramref name="headerCollection"/> ̂ꂩ
        ///  <c>null</c> QƂłB
        /// </exception>
        /// <exception cref="ArgumentException">
        /// ʐMŕKvȃf[^słB
        /// </exception>
        /// <exception cref="WebException">
        /// ʐMG[܂B܂́ANGXgLZ܂B
        /// </exception>
        /// <exception cref="CommunicationException">
        /// MɓŃG[܂B
        /// </exception>
        /// <param name="request">MsNGXgIuWFNgB</param>
        /// <param name="paramData">Mf[^i[IuWFNgB</param>
        /// <param name="headerCollection">M HTTP wb_Ƃėpwb_̃RNVB</param>
        /// <param name="reporter">is󋵒ʒms <see cref="IProgressChangeReporter"/> CX^XB</param>
        public virtual void Send(HttpWebRequest request,
                                 TParam paramData,
                                 IDictionary<string, string> headerCollection,
                                 IProgressChangeReporter reporter)
        {
            // ̓`FbN
            if (request == null)
            {
                ArgumentNullException exception = new ArgumentNullException("request");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(Properties.Resources.E_NULL_ARGUMENT, "request"), exception);
                }
                throw exception;

            }

            if (paramData == null)
            {
                ArgumentNullException exception = new ArgumentNullException("paramData");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(Properties.Resources.E_NULL_ARGUMENT, "paramData"), exception);
                }
                throw exception;
            }

            if (headerCollection == null)
            {
                ArgumentNullException exception = new ArgumentNullException("headerCollection");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(Properties.Resources.E_NULL_ARGUMENT, "headerCollection"), exception);
                }
                throw exception;
            }

            // NGXgf[^ҏW
            PrepareRequest(request, headerCollection);

            // Ms
            SendRequest(request, paramData, headerCollection, reporter);
        }


        /// <summary>
        /// <paramref name="request"/> œnꂽ <see cref="HttpWebRequest"/> pāA
        /// <paramref name="paramData"/>  <paramref name="headerCollection"/> ̏
        /// M܂B
        /// </summary>
        /// <remarks>
        /// <para>ʐMŔ <see cref="WebException"/> ̓Lb`ɃX[܂B܂AŗO
        /// ꍇAʏ <see cref="CommunicationException"/> ɃbvăX[悤ɂ܂BR[fBO
        /// ~Xɂ <see cref="ArgumentNullException"/>  <see cref="ArgumentException"/> At@CVXeݒ
        /// t@Cُ̈ɂOnhOKv͂܂B</para>
        /// <para>
        /// ̃\bhہA <paramref name="request"/>  <see cref="HttpWebRequest.GetRequestStream"/>
        /// \bh𗘗pď݃Xg[̎擾s܂B <see cref="HttpWebRequest.BeginGetRequestStream"/> 
        /// \bh𗘗pĔ񓯊݂sƂ͔ĂB񓯊ݏꍇɂ́Aǂݍ
        ///  <see cref="HttpWebRequest.BeginGetResponse"/> p񓯊ǂݍ݂ƂKv܂B
        /// </para>
        /// </remarks>
        /// <param name="request">MsNGXgIuWFNgB</param>
        /// <param name="paramData">Mf[^i[IuWFNgB</param>
        /// <param name="headerCollection">M HTTP wb_Ƃėpwb_̃RNVB</param>
        /// <param name="reporter">is󋵒ʒms <see cref="IProgressChangeReporter"/> CX^XB</param>
        protected abstract void SendRequest(HttpWebRequest request,
                                            TParam paramData,
                                            IDictionary<string, string> headerCollection,
                                            IProgressChangeReporter reporter);

        /// <summary>
        /// f[^Mɗp <see cref="HttpWebRequest"/> CX^X̐ݒs܂B
        /// </summary>
        /// <remarks>
        /// <para> HTTP \bh Content-Type ݒ肵܂B̑A<paramref name="requestHeaders"/>
        /// i[ꂽwb_NGXgwb_ƂĒǉ܂B</para>
        /// <para> <paramref name="requestHeaders"/>  "Method" L[܂ "Content-Type" L[݂ꍇłA
        ///  <see cref="SenderBase{TParam}.ContentType"/> vpeB܂ <see cref="SenderBase{TParam}.Method"/> 
        /// vpeBlԂꍇɂ́AvpeB̒lD悳܂B</para>
        /// </remarks>
        /// <param name="request">Mɗp <see cref="HttpWebRequest"/> CX^XB</param>
        /// <param name="requestHeaders">Mɗpwb_[̃RNVB</param>
        protected virtual void PrepareRequest(WebRequest request, IDictionary<string, string> requestHeaders)
        {
            // \bhƃReg^Cvݒ
            request.Method = Method;
            request.ContentType = ContentType;

            // NGXgwb_̐ݒ
            foreach (string key in requestHeaders.Keys)
            {
                request.Headers.Add(key, requestHeaders[key]);
            }
        }

        /// <summary>
        /// MTCỸp[Ze[WvZB
        /// </summary>
        /// <param name="sendSize">MTCY</param>
        /// <param name="totalSize">vTCY</param>
        /// <returns>p[Ze[W</returns>
        /// <exception cref="OverflowException">
        /// MTCỸp[Ze[W̌vZŃI[o[t[܂B
        /// </exception>
        protected virtual int CalcSendPercentage(long sendSize, long totalSize)
        {
            int percentage = 0;
            checked
            {
                percentage = (int)(sendSize * 50 / totalSize);
            }
            return percentage;
        }
    }
}
