// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Web;
using System.Web.Configuration;
using System.Configuration;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Common.Logging;
using System.Reflection;
using System.Web;

namespace TERASOLUNA.Fw.WebUT
{
    /// <summary>
    /// TERASOLUNA.Fw.Web.WebUtils のテスト クラスです。すべての
    /// TERASOLUNA.Fw.Web.WebUtils 単体テストをここに含めます
    /// </summary>  
    [TestClass()]
    public class WebUtilsTest
    {

        private TestContext testContextInstance;
        private const string QUERY_TOKEN_NAME = "__Token";

        private CustomErrorsMode _mode = CustomErrorsMode.RemoteOnly;
        private string _defaultRedirect = null;
        private CustomError _error = null; 

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
            System.Configuration.Configuration config = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);
            CustomErrorsSection section = (CustomErrorsSection)config.GetSection("system.web/customErrors");
            _error = section.Errors[0];
            _mode = section.Mode;
            _defaultRedirect = section.DefaultRedirect;
        }
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        [TestCleanup()]
        public void MyTestCleanup()
        {
            System.Configuration.Configuration config = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);
            CustomErrorsSection section = (CustomErrorsSection)config.SectionGroups["system.web"].Sections["customErrors"];
            section.Mode = _mode;
            section.DefaultRedirect = _defaultRedirect;
            if (section.Errors.Count == 0)
            {
                section.Errors.Add(_error);
            }
            config.Save();
            ConfigurationManager.RefreshSection("system.web/customErrors");
        }
        //
        #endregion


        /// <summary>
        /// Transit (string) のテスト
        /// </summary>
        /// <TestTarget>Transit (string)</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestTransit01()
        {
            string pageID = null;

            try
            {
                WebUtils.Transit(pageID);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("pageID", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "pageID"));
        }

        /// <summary>
        /// Transit (string) のテスト
        /// </summary>
        /// <TestTarget>Transit (string)</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>ArgumentExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestTransit02()
        {
            string pageID = "";

            try
            {
                WebUtils.Transit(pageID);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "pageID"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "pageID"));
        }


        /// <summary>
        /// Transit (string,IDictionary<string, string>()) のテスト
        /// </summary>
        /// <TestTarget>Transit (string,IDictionary<string, string>())</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestTransit03()
        {
            string resultID = "TERASOLUNA";
            IDictionary<string, string> queries = null;

            try
            {
                WebUtils.Transit(resultID, queries);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("queries", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "queries"));
        }


        /// <summary>
        /// Transit Transit (string,string,string) のテスト
        /// </summary>
        /// <TestTarget>Transit (string,string,string)</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestTransit04()
        {
            string resultID = "TERASOLUNAID";
            string queryKey = null;
            string queryValue = "TERASOLUNAValue";

            try
            {
                WebUtils.Transit(resultID, queryKey, queryValue);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("queryKey", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "queryKey"));
        }


        /// <summary>
        /// Transit (string,string,string) のテスト
        /// </summary>
        /// <TestTarget>Transit (string,string,string)</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>ArgumentExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestTransit05()
        {
            string resultID = "TERASOLUNAID";
            string queryKey = "";
            string queryValue = "TERASOLUNAValue";

            try
            {
                WebUtils.Transit(resultID, queryKey, queryValue);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "queryKey"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "queryKey"));
        }


        /// <summary>
        /// Transit (string,string,string) のテスト
        /// </summary>
        /// <TestTarget>Transit (string,string,string)</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        //[TestMethod()]
        public void TestTransit06()
        {
            string resultID = "TERASOLUNAID";
            string queryKey = "TERASOLUNAKey";
            string queryValue = null;

            try
            {
                WebUtils.Transit(resultID, queryKey, queryValue);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("queryValue", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "queryValue"));
        }


        /// <summary>
        /// Transit (string,string,string) のテスト
        /// </summary>
        /// <TestTarget>Transit (string,string,string)</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>ArgumentExceptionがスローされること</TestAssertion>
        //[TestMethod()]
        public void TestTransit07()
        {
            string resultID = "TERASOLUNAID";
            string queryKey = "TERASOLUNAKey";
            string queryValue = "";

            try
            {
                WebUtils.Transit(resultID, queryKey, queryValue);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "queryValue"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "queryValue"));
        }


        /// <summary>
        /// AppendParameter (Url, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (Url, string, string)</TestTarget>
        /// <TestCondition>第一引数が絶対パスであること</TestCondition>
        /// <TestAssertion>クエリ文字列を付加したURLが返却されること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter01()
        {
            Uri url = new Uri("http://www.test/Test/Test.aspx");
            string key = "aaa";
            string value = "bbb";

            string expected = url + "?" + key + "=" + value;
            string actual;

            actual = WebUtils.AppendParameter(url, key, value);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }


        /// <summary>
        /// AppendParameter (Url, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (Url, string, string)</TestTarget>
        /// <TestCondition>第一引数がクエリ文字列を含むことであること</TestCondition>
        /// <TestAssertion>クエリ文字列を追加したURLが返却されること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter02()
        {
            Uri url = new Uri("http://www.test/Test/Test.aspx?hoge=test");
            string key = "aaa";
            string value = "bbb";

            string expected = url + "&" + key + "=" + value;
            string actual;
            actual = WebUtils.AppendParameter(url, key, value);
            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }


        /// <summary>
        /// AppendParameter (Uri, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第一引数に null が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentNullException スローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter03()
        {
            Uri url = null;
            string key = "aaa";
            string value = "bbb";
            string actual;

            try
            {
                actual = WebUtils.AppendParameter(url, key, value);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("url", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "url"));
        }


        /// <summary>
        /// AppendParameter (Uri, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第二引数に空文字が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentNullExceptionLスローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter04()
        {
            Uri url = new Uri("http://www.test/Test/Test.aspx");
            string key = "";
            string value = "bbb";
            string actual;

            try
            {
                actual = WebUtils.AppendParameter(url, key, value);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "key"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "key"));
        }
        /// <summary>
        /// AppendParameter (Uri, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第二引数にnullが指定されていること</TestCondition>
        /// <TestAssertion>ArgumentNullException スローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter05()
        {
            Uri url = new Uri("http://www.test/Test/Test.aspx");
            string key = null;
            string value = "bbb";
            string actual;

            try
            {
                actual = WebUtils.AppendParameter(url, key, value);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("key", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "key"));
        }

        /// <summary>
        /// AppendParameter (Uri, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第三引数に空文字が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentNullExceptionLスローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter06()
        {
            Uri url = new Uri("http://www.test/Test/Test.aspx");
            string expectedUrl = "http://www.test/Test/Test.aspx?aaa=";
            string key = "aaa";
            string value = "";
            string actual;

            actual = WebUtils.AppendParameter(url, key, value);
            
            // 確認
            Assert.AreEqual(expectedUrl, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }

        /// <summary>
        /// AppendParameter (Uri, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第三引数に null が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentNullExceptionLスローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter07()
        {
            Uri url = new Uri("http://www.test/Test/Test.aspx");
            string expectedUrl = url.ToString() + "?aaa=";
            string key = "aaa";
            string value = null;
            string actual;

            // 実施
            actual = WebUtils.AppendParameter(url, key, value);

            // 確認
            Assert.AreEqual(expectedUrl, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }


        /// <summary>
        /// AppendParameter (string, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第一引数に「?」が含まれないこと</TestCondition>
        /// <TestAssertion>クエリ文字列を付加したURLが返却されること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter08()
        {
            string url = "http://www.test/Test/Test.aspx";
            string key = "aaa";
            string value = "bbb";

            string expected = url + "?" + key + "=" + value;
            string actual;

            actual = WebUtils.AppendParameter(url, key, value);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }

        /// <summary>
        /// AppendParameter (string, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第一引数に「?」が含まれること</TestCondition>
        /// <TestAssertion>トークンのクエリ文字列を付加したURLが返却されること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter09()
        {
            string url = "http://www.test/Test/Test.aspx?hoge=test";
            string key = "aaa";
            string value = "bbb";

            string expected = url + "&" + key + "=" + value;
            string actual;

            actual = WebUtils.AppendParameter(url, key, value);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }
        /// <summary>
        /// AppendParameter (string, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第一引数に空文字列が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentExceptionLスローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter10()
        {
            string url = "";
            string key = "aaa";
            string value = "bbb";
            string actual;

            try
            {
                actual = WebUtils.AppendParameter(url, key, value);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "url"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "url"));
        }
        /// <summary>
        /// AppendParameter (string, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第一引数にnullが指定されていること</TestCondition>
        /// <TestAssertion>ArgumentNullExceptionLスローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter11()
        {
            string url = null;
            string key = "aaa";
            string value = "bbb";
            string actual;

            try
            {
                actual = WebUtils.AppendParameter(url, key, value);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("url", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "url"));
        }

        /// <summary>
        /// AppendParameter (string, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第一引数に空文字列が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentExceptionLスローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter12()
        {
            string url = "http://www.test/Test/Test.aspx";
            string key = "";
            string value = "bbb";
            string actual;

            try
            {
                actual = WebUtils.AppendParameter(url, key, value);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "key"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "key"));
        }

        /// <summary>
        /// AppendParameter (string, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第二引数にnullが指定されていること</TestCondition>
        /// <TestAssertion>ArgumentNullExceptionLスローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter13()
        {
            string url = "http://www.test/Test/Test.aspx";
            string key = null;
            string value = "bbb";
            string actual;

            try
            {
                actual = WebUtils.AppendParameter(url, key, value);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("key", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "key"));
        }

        /// <summary>
        /// AppendParameter (string, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第二引数に空文字列が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentExceptionLスローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter14()
        {
            string url = "http://www.test/Test/Test.aspx";
            string expectedUrl = "http://www.test/Test/Test.aspx?aaa=";
            string key = "aaa";
            string value = "";
            string actual;

            actual = WebUtils.AppendParameter(url, key, value);

            // 確認
            Assert.AreEqual(expectedUrl, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }

        /// <summary>
        /// AppendParameter (string, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (string, string, string)</TestTarget>
        /// <TestCondition>第三引数に null が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentExceptionLスローされること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter15()
        {
            string url = "http://www.test/Test/Test.aspx";
            string expectedUrl = url + "?aaa=";
            string key = "aaa";
            string value = null;
            string actual;

            // 実施
            actual = WebUtils.AppendParameter(url, key, value);
            
            // 確認
            Assert.AreEqual(expectedUrl, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }
        /// <summary>
        /// AppendParameter (Uri, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (Uri, string, string)</TestTarget>
        /// <TestCondition>Uri が Test/Test.aspx</TestCondition>
        /// <TestAssertion>クエリ文字列が追加されること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter16()
        {
            Uri url = new Uri("Test/Test.aspx", UriKind.Relative);
            string key = "aaa";
            string value = "bbb";

            string expected = url + "?" + key + "=" + value;
            string actual;

            actual = WebUtils.AppendParameter(url, key, value);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }
        /// <summary>
        /// AppendParameter (Uri, string, string) のテスト
        /// </summary>
        /// <TestTarget>AppendParameter (Uri, string, string)</TestTarget>
        /// <TestCondition>Uri が Test/Test.aspx?hoge=test</TestCondition>
        /// <TestAssertion>クエリ文字列が追加されること</TestAssertion>
        [TestMethod()]
        public void TestAppendParameter17()
        {
            Uri url = new Uri("Test/Test.aspx?hoge=test", UriKind.Relative);
            string key = "aaa";
            string value = "bbb";

            string expected = url + "&" + key + "=" + value;
            string actual;
            actual = WebUtils.AppendParameter(url, key, value);
            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }
        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイルにcustomErrors要素が設定されていること
        /// 第二引数にcustomErrors要素のdefaultRedirect属性に対応するパスを指定すること
        /// </TestCondition>
        /// <TestAssertion>trueが返却されること</TestAssertion>
        [DeploymentItem("TERASOLUNA.Fw.Web.dll")]
        [TestMethod()]
        public void TestIsInCustomErrorPages01()
        {
            string appPath = "/appRoot";
            string path = "/appRoot/ErrorDefault.aspx";

            bool actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);

            Assert.IsTrue(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));

        }

        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイルにcustomErrors要素が設定されていること
        /// 第二引数にerror要素のredirect属性に対応するパスを指定すること
        /// </TestCondition>
        /// <TestAssertion>trueが返却されること</TestAssertion>
        [TestMethod()]
        public void TestIsInCustomErrorPages02()
        {
            string appPath = "/appRoot";
            string path = "/appRoot/Error404.aspx";

            bool actual;

            actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);

            Assert.IsTrue(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }

        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイルにcustomErrors要素が設定されていること
        /// 第二引数にcustomErrorsに設定されていないパスを指定すること
        /// </TestCondition>
        /// <TestAssertion>falseが返却されること</TestAssertion>
        [TestMethod()]
        public void TestIsInCustomErrorPages03()
        {
            string appPath = "/appRoot";
            string path = "/appRoot/Nothing.aspx";

            bool actual;

            actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);

            Assert.IsFalse(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }

        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイルにcustomErrors要素が設定されていること
        /// 第一引数にnullが指定されていること
        /// </TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestIsInCustomErrorPages04()
        {
            string appPath = null;
            string path = "/appRoot/Nothing.aspx";
            bool actual;

            try
            {
                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                ArgumentNullException exception = e.InnerException as ArgumentNullException;
                Assert.AreEqual("appPath", exception.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "appPath"));
        }

        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイルにcustomErrors要素が設定されていること
        /// 第一引数に空文字列が指定されていること
        /// </TestCondition>
        /// <TestAssertion>ArgumentExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestIsInCustomErrorPages05()
        {
            string appPath = "";
            string path = "/appRoot/Nothing.aspx";
            bool actual;

            try
            {
                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                ArgumentException exception = e.InnerException as ArgumentException;
                Assert.AreEqual(typeof(ArgumentException).Name, exception.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "appPath"), exception.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "appPath"));
        }

        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイルにcustomErrors要素が設定されていること
        /// 第二引数にnullが指定されていること
        /// </TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestIsInCustomErrorPages06()
        {
            string appPath = "/appRoot";
            string path = null;
            bool actual;

            try
            {
                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                ArgumentNullException exception = e.InnerException as ArgumentNullException;
                Assert.AreEqual("path", exception.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "path"));
        }

        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイルにcustomErrors要素が設定されていること
        /// 第二引数に空文字列が指定されていること
        /// </TestCondition>
        /// <TestAssertion>ArgumentExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestIsInCustomErrorPages07()
        {
            string appPath = "/appRoot";
            string path = "";

            bool actual;

            try
            {
                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                ArgumentException exception = e.InnerException as ArgumentException;
                Assert.AreEqual(typeof(ArgumentException).Name, exception.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "path"), exception.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "path"));
        }

        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイルにcustomErrors要素が設定されていること
        /// </TestCondition>
        /// <TestAssertion>falseが返却されること</TestAssertion>
        [TestMethod()]
        public void TestIsInCustomErrorPages08()
        {
            string appPath = "/appRoot";
            string path = "/appRoot/ErrorDefault.aspx";

            bool actual;

            System.Configuration.Configuration config = null;
            CustomErrorsSection section = null;
            CustomErrorsMode mode = CustomErrorsMode.On;
            string defaultRedirect = null;
            CustomError[] elements = null;

            try
            {
                config = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);
                section = (CustomErrorsSection)config.SectionGroups["system.web"].Sections["customErrors"];
                mode = section.Mode;
                defaultRedirect = section.DefaultRedirect;
                elements = new CustomError[1];
                section.Errors.CopyTo(elements, 0);

                config.SectionGroups["system.web"].Sections.Remove("customErrors");
                config.Save();
                ConfigurationManager.RefreshSection("system.web/customErrors");

                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);

                Assert.IsFalse(actual);
            }
            finally
            {
                if (config != null && section != null)
                {
                    config = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);
                    section = (CustomErrorsSection)config.SectionGroups["system.web"].Sections["customErrors"];
                    section.Mode = mode;
                    section.DefaultRedirect = defaultRedirect;
                    section.Errors.Add(elements[0]);
                    config.Save();
                    ConfigurationManager.RefreshSection("system.web/customErrors");
                }
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }

        /// <summary>
        /// GetCurrentPagePath (string, string) のテスト
        /// </summary>
        /// <TestTarget>GetCurrentPagePath (string, string)</TestTarget>
        /// <TestCondition>引数に正常な値が指定されていること</TestCondition>
        /// <TestAssertion>第二引数から第一引数を取り除いた文字列が返却されること</TestAssertion>
        [TestMethod()]
        public void TestGetCurrentPagePath01()
        {
            string appPath = "/appRoot";
            string filePath = "/appRoot/Test.aspx";

            string expected = "/Test.aspx";
            string actual;

            actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._GetCurrentPagePath(appPath, filePath);

            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }

        /// <summary>
        /// GetCurrentPagePath (string, string) のテスト
        /// </summary>
        /// <TestTarget>GetCurrentPagePath (string, string)</TestTarget>
        /// <TestCondition>第一引数にnullが指定されていること</TestCondition>
        /// <TestAssertion>ArgumentNullExceptionが返却されること</TestAssertion>
        [TestMethod()]
        public void TestGetCurrentPagePath02()
        {
            string appPath = null;
            string filePath = "/appRoot/Test.aspx";
            string actual;

            try
            {
                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._GetCurrentPagePath(appPath, filePath);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                ArgumentNullException exception = e.InnerException as ArgumentNullException;
                Assert.AreEqual("appPath", exception.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "appPath"));
        }

        /// <summary>
        /// GetCurrentPagePath (string, string) のテスト
        /// </summary>
        /// <TestTarget>GetCurrentPagePath (string, string)</TestTarget>
        /// <TestCondition>第一引数に空文字列が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentExceptionが返却されること</TestAssertion>
        [TestMethod()]
        public void TestGetCurrentPagePath03()
        {
            string appPath = "";
            string filePath = "/appRoot/Test.aspx";
            string actual;

            try
            {
                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._GetCurrentPagePath(appPath, filePath);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                ArgumentException exception = e.InnerException as ArgumentException;
                Assert.AreEqual(typeof(ArgumentException).Name, exception.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "appPath"), exception.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "appPath"));
        }

        /// <summary>
        /// GetCurrentPagePath (string, string) のテスト
        /// </summary>
        /// <TestTarget>GetCurrentPagePath (string, string)</TestTarget>
        /// <TestCondition>第二引数にnullが指定されていること</TestCondition>
        /// <TestAssertion>ArgumentNullExceptionが返却されること</TestAssertion>
        [TestMethod()]
        public void TestGetCurrentPagePath04()
        {
            string appPath = "/appRoot";
            string filePath = null;
            string actual;

            try
            {
                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._GetCurrentPagePath(appPath, filePath);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                ArgumentNullException exception = e.InnerException as ArgumentNullException;
                Assert.AreEqual("filePath", exception.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "filePath"));
        }

        /// <summary>
        /// GetCurrentPagePath (string, string) のテスト
        /// </summary>
        /// <TestTarget>GetCurrentPagePath (string, string)</TestTarget>
        /// <TestCondition>第二引数に空文字列が指定されていること</TestCondition>
        /// <TestAssertion>ArgumentExceptionが返却されること</TestAssertion>
        [TestMethod()]
        public void TestGetCurrentPagePath05()
        {
            string appPath = "/appRoot";
            string filePath = "";
            string actual;

            try
            {
                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._GetCurrentPagePath(appPath, filePath);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                ArgumentException exception = e.InnerException as ArgumentException;
                Assert.AreEqual(typeof(ArgumentException).Name, exception.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "filePath"), exception.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException), "E_EMPTY_STRING", "filePath"));
        }

        /// <summary>
        /// GetCurrentPagePath (string, string) のテスト
        /// </summary>
        /// <TestTarget>GetCurrentPagePath (string, string)</TestTarget>
        /// <TestCondition>第一引数の文字数が第二引数の文字数より多いこと</TestCondition>
        /// <TestAssertion>ArgumentExceptionが返却されること</TestAssertion>
        [TestMethod()]
        public void TestGetCurrentPagePath06()
        {
            string appPath = "/appRootaaaaaaaaaaa";
            string filePath = "/appRoot/Test.aspx";
            string actual;

            try
            {
                actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._GetCurrentPagePath(appPath, filePath);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                ArgumentException exception = e.InnerException as ArgumentException;
                Assert.AreEqual(typeof(ArgumentException).Name, exception.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_FILE_PATH", appPath, filePath), exception.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(WebUtils), typeof(ArgumentException),"E_INVALID_FILE_PATH", appPath, filePath));
        }

        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>defaultRedirect属性の指定がないこと</TestCondition>
        /// <TestAssertion>falseが返却されること</TestAssertion>
        [DeploymentItem("TERASOLUNA.Fw.Web.dll")]
        [TestMethod()]
        public void TestIsInCustomErrorPages09()
        {
            string appPath = "/appRoot";
            string path = "/appRoot/ErrorDefault.aspx";

            bool actual;

            System.Configuration.Configuration config = null;
            CustomErrorsSection section = null;
            CustomErrorsMode mode = CustomErrorsMode.On;
            string defaultRedirect = null;

            config = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);
            section = (CustomErrorsSection)config.SectionGroups["system.web"].Sections["customErrors"];
            mode = section.Mode;
            defaultRedirect = section.DefaultRedirect;
            section.DefaultRedirect = "";

            config.Save();
            ConfigurationManager.RefreshSection("system.web/customErrors");

            actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);

            Assert.IsFalse(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }

        /// <summary>
        /// IsInCustomErrorPages (string, string) のテスト
        /// </summary>
        /// <TestTarget>IsInCustomErrorPages (string, string)</TestTarget>
        /// <TestCondition>defaultRedirect属性の指定がないこと</TestCondition>
        /// <TestAssertion>falseが返却されること</TestAssertion>
        [DeploymentItem("TERASOLUNA.Fw.Web.dll")]
        [TestMethod()]
        public void TestIsInCustomErrorPages10()
        {
            string appPath = "/appRoot";
            string path = "/appRoot/ErrorDefault.aspx";

            bool actual;

            System.Configuration.Configuration config = null;
            CustomErrorsSection section = null;
            CustomErrorsMode mode = CustomErrorsMode.On;
            string defaultRedirect = null;
            CustomError[] elements = null;

            config = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);
            section = (CustomErrorsSection)config.SectionGroups["system.web"].Sections["customErrors"];
            mode = section.Mode;
            defaultRedirect = section.DefaultRedirect;
            section.DefaultRedirect = "";
            elements = new CustomError[1];
            section.Errors.CopyTo(elements, 0);

            section.Errors.Remove("404");

            config.Save();
            ConfigurationManager.RefreshSection("system.web/customErrors");

            actual = TERASOLUNA_Fw_Web_WebUtilsAccessor._IsInCustomErrorPages(appPath, path);
            Assert.IsFalse(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(WebUtils), LogLevel.INFO));
        }
    }


}

