// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Text.RegularExpressions;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Common.Validation
{
    /// <summary>
    /// Validation Application Block 𗘗p͒l؃NXpA
    /// `FbN񋟂郆[eBeBNXłB
    /// </summary>
    public static class CaseCheckUtil
    {
        /// <summary>
        /// <see cref="ILog"/> NX̃CX^XłB
        /// </summary>
        /// <remarks>
        /// Oo͂ɗp܂B
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(CaseCheckUtil));

        /// <summary>
        /// pJĩ`FbNɎgp镶łB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "¯֬ܦް" łB
        /// </remarks>
        private static readonly string HANKAKU_KANA_LIST =
            "¯֬ܦް";

        ///<summary>
        /// SpJĩ`FbNɎgp镶łB
        ///</summary>
        /// <remarks>
        /// ̒萔̒l "ACEGI@BDFHJLNPRKMOQSTVXZ\UWY[]^`ceg_adfhijklmnqtwzorux{psvy|}~b[" łB
        /// </remarks>
        private static readonly string ZENKAKU_KANA_LIST =
            "ACEGI@BDFHJLNPRKMOQSTVXZ\"
                + "UWY[]^`ceg_adfhijklmnqtwz"
                + "orux{psvy|}~"
                + "b[";

        ///<summary>
        /// pp̃`FbNɎgp鐳K\łB
        ///</summary>
        private static readonly Regex ALPHA_NUMERIC_REGEXP =
                new Regex(@"^([0-9]|[a-z]|[A-Z])*$");

        ///<summary>
        /// pp啶̃`FbNɎgp鐳K\łB
        ///</summary>
        private static readonly Regex ALPHA_UPPER_NUMERIC_REGEXP =
                new Regex(@"^([0-9]|[A-Z])*$");

        ///<summary>
        /// l̃`FbNɎgp鐳K\łB
        ///</summary>
        private static readonly Regex NUMERIC_STRING = new Regex("^([0-9])*$");

        /// <summary>
        /// UNICODER[h'&#165;u0000''&#165;u00ff'
        /// ͈͓ɑ݂SpłB
        /// </summary>
        private static readonly string ZENKAKU_BEGIN_U00_LIST =
            "_Nʁ}L~";

        /// <summary>
        /// w肳ꂽpJił邱Ƃ`FbN܂B
        /// </summary>
        /// <param name="c">`FbNΏۂ̕B</param>
        /// <returns>pJił true BłȂ false B</returns>
        private static bool IsHankakuKanaChar(char c)
        {
            return HANKAKU_KANA_LIST.IndexOf(c) >= 0;
        }

        /// <summary>
        /// w肳ꂽpł邱Ƃ`FbN܂B
        /// </summary>
        /// <param name="c">`FbNΏۂ̕B</param>
        /// <returns>pł true BłȂ false B</returns>
        ///
        private static bool IsHankakuChar(char c)
        {
            return (c <= '\u00ff' && ZENKAKU_BEGIN_U00_LIST.IndexOf(c) < 0)
                    || IsHankakuKanaChar(c);
        }

        /// <summary>
        /// w肳ꂽSpł邱Ƃ`FbN܂B
        /// </summary>
        /// <param name="c">`FbNΏۂ̕B</param>
        /// <returns>Spł true BłȂ false B</returns>
        private static bool IsZenkakuChar(char c)
        {
            return !IsHankakuChar(c);
        }

        /// <summary>
        /// w肳ꂽSpJił邱Ƃ`FbNB
        /// </summary>
        /// <param name="c">`FbNΏۂ̕B</param>
        /// <returns>SpJił true BłȂ false B</returns>
        private static bool IsZenkakuKanaChar(char c)
        {
            return ZENKAKU_KANA_LIST.IndexOf(c) >= 0;
        }

        /// <summary>
        /// ؒlpp݂̂ō\镶ł邱Ƃ`FbN܂B
        /// </summary>
        /// <remarks> 
        /// ppƂ́Ap( 0 ` 9 )Ɣpp( a ` z A A ` Z )𕹂WłB
        /// </remarks>
        /// <param name="value">ؒlB</param>
        /// <returns>ؒlpp݂̂ō\镶A܂͋󕶎̏ꍇA<c>true</c> BȊȌꍇA<c>false</c> B</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="value"/>  null QƂłB
        /// </exception>
        public static bool IsAlphaNumericString(string value)
        {
            if (value == null)
            {
                ArgumentNullException exception = new ArgumentNullException("value");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "value"), exception);
                }
                throw exception;
            }

            if (value.Length == 0)
            {
                return true;
            }
            return ALPHA_NUMERIC_REGEXP.IsMatch(value);
        }

        /// <summary>
        /// ؒlpp啶݂̂ō\镶ł邱Ƃ`FbN܂B
        /// </summary>
        /// <remarks>
        /// pp啶Ƃ́Ap( 0 ` 9 )Ɣp啶̉p( A ` Z )𕹂WłB
        /// </remarks>
        /// <param name="value">ؒlB</param>
        /// <returns>ؒlpp啶݂̂ō\镶A܂͋󕶎̏ꍇA<c>true</c> BȊȌꍇA<c>false</c> B</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="value"/>  null QƂłB
        /// </exception>
        public static bool IsUpperAlphaNumericString(string value)
        {
            if (value == null)
            {
                ArgumentNullException exception = new ArgumentNullException("value");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "value"), exception);
                }
                throw exception;
            }

            if (value.Length == 0)
            {
                return true;
            }
            return ALPHA_UPPER_NUMERIC_REGEXP.IsMatch(value);
        }

        /// <summary>
        /// ؒll݂̂ō\镶ł邱Ƃ`FbN܂B
        /// </summary>
        /// <remarks>
        /// lƂ́Ap( 0 ` 9 )łB
        /// </remarks>
        /// <param name="value">ؒlB</param>
        /// <returns>ؒll݂̂ō\镶A܂͋󕶎̏ꍇA<c>true</c> BȊȌꍇA<c>false</c> B</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="value"/>  null QƂłB
        /// </exception>
        public static bool IsNumericString(string value)
        {
            if (value == null)
            {
                ArgumentNullException exception = new ArgumentNullException("value");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "value"), exception);
                }
                throw exception;
            }

            if (value.Length == 0)
            {
                return true;
            }

            return NUMERIC_STRING.IsMatch(value);
        }

        /// <summary>
        /// ؒlpJî݂ō\镶ł邱Ƃ`FbN܂B
        /// </summary>
        /// <remarks>
        /// pJiƂ́AɎWłB"¯֬ܦް"
        /// </remarks>
        /// <param name="value">ؒlB</param>
        /// <returns>ؒlpp݂̂ō\镶A܂͋󕶎̏ꍇA<c>true</c> BȊȌꍇA<c>false</c> B</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="value"/>  null QƂłB
        /// </exception>
        public static bool IsHankakuKanaString(string value)
        {
            // ؒlnull܂͋󕶎̎Atrueԋp
            if (value == null)
            {
                ArgumentNullException exception = new ArgumentNullException("value");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "value"), exception);
                }
                throw exception;
            }

            if (value.Length == 0)
            {
                return true;
            }

            char[] chars = value.ToCharArray();
            for (int i = 0; i < chars.Length; i++)
            {
                if (!IsHankakuKanaChar(chars[i]))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// ؒlp݂̂ō\镶ł邱Ƃ`FbN܂B
        /// </summary>
        /// <remarks>
        /// pƂ́A"_Nʁ}L~" unicode 0255Ԗڂ܂(g ASCII R[h͈̔)̕ɔpJiWłB
        /// </remarks>
        /// <param name="value">ؒlB</param>
        /// <returns>ؒlp݂̂ō\镶A܂͋󕶎̏ꍇA<c>true</c> BȊȌꍇA<c>false</c> B</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="value"/>  null QƂłB
        /// </exception>
        public static bool IsHankakuString(string value)
        {
            // ؒlnull܂͋󕶎̎Atrueԋp
            if (value == null)
            {
                ArgumentNullException exception = new ArgumentNullException("value");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "value"), exception);
                }
                throw exception;
            }

            if (value.Length == 0)
            {
                return true;
            }

            char[] chars = value.ToCharArray();
            for (int i = 0; i < chars.Length; i++)
            {
                if (!IsHankakuChar(chars[i]))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// ؒlSp݂̂ō\镶ł邱Ƃ`FbN܂B
        /// </summary>
        /// <remarks>
        /// SpƂ́Ap(p`FbN̒`ɏ)ł͂ȂWłB
        /// </remarks>
        /// <param name="value">ؒlB</param>
        /// <returns>ؒlSp݂̂ō\镶A܂͋󕶎̏ꍇA<c>true</c> BȊȌꍇA<c>false</c> B</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="value"/>  null QƂłB
        /// </exception>
        public static bool IsZenkakuString(string value)
        {
            // ؒlnull܂͋󕶎̎Atrueԋp
            if (value == null)
            {
                ArgumentNullException exception = new ArgumentNullException("value");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "value"), exception);
                }
                throw exception;
            }

            if (value.Length == 0)
            {
                return true;
            }

            char[] chars = value.ToCharArray();
            for (int i = 0; i < chars.Length; i++)
            {
                if (!IsZenkakuChar(chars[i]))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// ؒlSpJî݂ō\镶ł邱Ƃ`FbN܂B
        /// </summary>
        /// <remarks>
        /// SpJiƂ́AɎWłB 
        /// <para>
        /// "ACEGI@BDFHJLNPRKMOQSTVXZ\UWY[]
        /// ^`ceg_adfhijklmnqtwzorux{psvy|}~
        /// "b["
        /// </para>
        /// </remarks>
        /// <param name="value">ؒlB</param>
        /// <returns>ؒlSpJî݂ō\镶A܂͋󕶎̏ꍇA<c>true</c> BȊȌꍇA<c>false</c> B</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="value"/>  null QƂłB
        /// </exception>
        public static bool IsZenkakuKanaString(string value)
        {
            // ؒlnull܂͋󕶎̎Atrueԋp
            if (value == null)
            {
                ArgumentNullException exception = new ArgumentNullException("value");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "value"), exception);
                }
                throw exception;
            }

            if (value.Length == 0)
            {
                return true;
            }

            char[] chars = value.ToCharArray();
            for (int i = 0; i < chars.Length; i++)
            {
                if (!IsZenkakuKanaChar(chars[i]))
                {
                    return false;
                }
            }
            return true;
        }
    }
}
