﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Configuration.Conversion;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Client.Configuration;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
namespace TERASOLUNA.Fw.ClientUT.Configuration.Conversion
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Configuration.Conversion.ConversionConfiguration のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Configuration.Conversion.ConversionConfiguration 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class ConversionConfigurationTest
    {

        private TestContext testContextInstance;

        ///<summary>
        /// 読み込む設定ファイルです。
        ///</summary>
        private const string CONFIG_FILE = "ConversionConfiguration_Section01.config";

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
            PrivateType config = new PrivateType(typeof(TerasolunaClientConfiguration));
            ConversionConfigurationSection section = ConfigTestHelper.GetConversionConfigurationSectionFromCustomFile(CONFIG_FILE);
            section.CreateConvertElements();
            config.SetStaticField("_conversionConfigSection", section);
        }

        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        [TestCleanup()]
        public void MyTestCleanup()
        {
            PrivateType config = new PrivateType(typeof(TerasolunaClientConfiguration));
            config.SetStaticField("_conversionConfigSection", null);
        }
        //
        #endregion


        ///<summary>
        /// GetConvert (string) のテスト
        ///</summary>
        ///<remarks>
        /// 存在するConverter IDを指定した場合、Convert要素が取得できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetConvertTest01()
        {
            ConvertElement actual = ConversionConfiguration.GetConvert("getDest");
            Assert.IsNotNull(actual);
            Assert.AreEqual(1, actual.Param.Columns.Count);
            Assert.AreEqual(1, actual.Result.Columns.Count);
            Assert.AreEqual(1, actual.Result.ClearViews.Count);
            Assert.AreEqual("Info.DestCode", actual.Param.Columns[0].SrcName);
            Assert.AreEqual("DtoDest.DestCode", actual.Param.Columns[0].DestName);
            Assert.AreEqual("Info.DestName", actual.Result.Columns[0].SrcName);
            Assert.AreEqual("DtoDest.DestName", actual.Result.Columns[0].DestName);
            Assert.AreEqual("Info", actual.Result.ClearViews[0].TableName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));

        }

        ///<summary>
        /// GetConvert (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数にnullを渡した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetConvertTest02()
        {
            string id = null;

            try
            {
                ConvertElement actual = ConversionConfiguration.GetConvert(id);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("id", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConversionConfiguration), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "id"));

        }

        ///<summary>
        /// GetConvert (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数に空文字を渡した場合、ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetConvertTest03()
        {
            string id = string.Empty;

            try
            {
                ConvertElement actual = ConversionConfiguration.GetConvert(id);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "id"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConversionConfiguration), typeof(ArgumentException), "E_EMPTY_STRING", "id"));

        }

        ///<summary>
        /// GetConvert (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数に存在しないConvert Idを渡した場合、nullが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetConvertTest04()
        {
            ConvertElement actual = ConversionConfiguration.GetConvert("Nothing");
            Assert.IsNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));

        }

        ///<summary>
        /// GetParam (string) のテスト
        ///</summary>
        ///<remarks>
        /// 存在するConverter IDを指定した場合、Param要素が取得できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetParamTest01()
        {
            ParamElement actual = ConversionConfiguration.GetParam("getDest");
            Assert.IsNotNull(actual);
            Assert.AreEqual(1, actual.Columns.Count);
            Assert.AreEqual("Info.DestCode", actual.Columns[0].SrcName);
            Assert.AreEqual("DtoDest.DestCode", actual.Columns[0].DestName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));

        }

        ///<summary>
        /// GetParam (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数にnullを指定した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetParamTest02()
        {
            string id = null;

            try
            {
                ParamElement actual = ConversionConfiguration.GetParam(id);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("id", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConversionConfiguration), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "id"));

        }

        ///<summary>
        /// GetParam (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数に空文字を指定した場合、ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetParamTest03()
        {
            string id = string.Empty;

            try
            {
                ParamElement actual = ConversionConfiguration.GetParam(id);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "id"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConversionConfiguration), typeof(ArgumentException), "E_EMPTY_STRING", "id"));

        }

        ///<summary>
        /// GetParam (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数に存在しないIDを指定した場合、TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void GetParamTest04()
        {
            string id = "Nothing";

            try
            {
                ParamElement actual = ConversionConfiguration.GetParam(id);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TYPE_NOT_FOUND", id), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConversionConfiguration), typeof(TerasolunaException), "E_CONVERSION_TYPE_NOT_FOUND", id));

        }

        ///<summary>
        /// GetResult (string) のテスト
        ///</summary>
        ///<remarks>
        /// 存在するIDを指定した場合、Result要素が返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetResultTest01()
        {
            ResultElement actual = ConversionConfiguration.GetResult("getDest");
            Assert.IsNotNull(actual);
            Assert.AreEqual(1, actual.Columns.Count);
            Assert.AreEqual(1, actual.ClearViews.Count);
            Assert.AreEqual("Info.DestName", actual.Columns[0].SrcName);
            Assert.AreEqual("DtoDest.DestName", actual.Columns[0].DestName);
            Assert.AreEqual("Info", actual.ClearViews[0].TableName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));

        }

        ///<summary>
        /// GetResult (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数にnullを指定した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetResultTest02()
        {
            try
            {
                ResultElement actual = ConversionConfiguration.GetResult(null);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("id", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConversionConfiguration), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "id"));
        }

        ///<summary>
        /// GetResult (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数に空文字を指定した場合、ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetResultTest03()
        {
            try
            {
                ResultElement actual = ConversionConfiguration.GetResult(string.Empty);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "id"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConversionConfiguration), typeof(ArgumentException), "E_EMPTY_STRING", "id"));

        }

        ///<summary>
        /// GetResult (string) のテスト
        ///</summary>
        ///<remarks>
        /// 存在しないIDを指定した場合、TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_Section01.config")]
        [DeploymentItem(@"Configuration\Conversion\ConversionConfigurationTestData\ConversionConfiguration_External01.config")]
        public void GetResultTest04()
        {
            string id = "Nothing";

            try
            {
                ResultElement actual = ConversionConfiguration.GetResult(id);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TYPE_NOT_FOUND", id), e.Message);
            }

            //// ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConversionConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConversionConfiguration), typeof(TerasolunaException), "E_CONVERSION_TYPE_NOT_FOUND", id));
        }
    }
}


