// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Communication;
using TERASOLUNA.Fw.ClientUT.Communication.SenderBaseData;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Reflection;

namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.Sender.SenderBase&lt;T&gt; のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Communication.Sender.SenderBase&lt;T&gt; 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class SenderBaseTest
    {
        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        /// Method のテスト
        /// </summary>
        /// <remarks>
        /// 必ず "POST" を取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void MethodTest01()
        {
            // 準備
            SenderBase<string> target = new SenderBaseImpl();
            string expectedPostMethodName = "POST";

            // 実行
            string actualMethod = target.Method;

            // 結果
            Assert.AreEqual(expectedPostMethodName, actualMethod);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<>), LogLevel.INFO));
        }

        /// <summary>
        /// BufferSize のテスト
        /// </summary>
        /// <remarks>
        /// 1 以上の数値を設定した場合、
        /// 設定した値を取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void BufferSizeTest01()
        {
            // 準備
            SenderBase<string> target = new SenderBaseImpl();
            int expectedBufferSize = 1;

            // 実行
            target.BufferSize = expectedBufferSize;
            int actualBufferSize = target.BufferSize;

            // 結果
            Assert.AreEqual(expectedBufferSize, actualBufferSize);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<>), LogLevel.INFO));
        }

        /// <summary>
        /// BufferSize のテスト
        /// </summary>
        /// <remarks>
        /// 0 を設定した場合、
        /// デフォルトの BufferSize 値を取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void BufferSizeTest02()
        {
            // 準備
            SenderBase<string> target = new SenderBaseImpl();
            int expectedBufferSize = 0;
            int expectedDefaultBufferSize = 1024 * 8;

            // 実行
            target.BufferSize = expectedBufferSize;
            int actualBufferSize = target.BufferSize;

            // 結果
            Assert.AreEqual(expectedDefaultBufferSize, actualBufferSize);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<>), LogLevel.INFO));
        }

        /// <summary>
        /// BufferSize のテスト
        /// </summary>
        /// <remarks>
        /// -1 を設定した場合、
        /// デフォルトの BufferSize 値を取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void BufferSizeTest03()
        {
            // 準備
            SenderBase<string> target = new SenderBaseImpl();
            int expectedBufferSize = -1;
            int expectedDefaultBufferSize = 1024 * 8;

            // 実行
            target.BufferSize = expectedBufferSize;
            int actualBufferSize = target.BufferSize;

            // 結果
            Assert.AreEqual(expectedDefaultBufferSize, actualBufferSize);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<>), LogLevel.INFO));
        }

        /// <summary>
        /// CalcSendPercentage (int, long) のテスト
        /// </summary>
        [TestMethod()]
        public void CalcSendPercentageTest01()
        {
            SenderBase<string> target = new SenderBaseImpl();
            TERASOLUNA_Fw_Client_Communication_SenderBaseImplAccessor accessor = new TERASOLUNA_Fw_Client_Communication_SenderBaseImplAccessor(target);
            int actual = accessor.CalcSendPercentage(0, 1);

            Assert.AreEqual(0, actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<>), LogLevel.INFO));
        }

        /// <summary>
        /// CalcSendPercentage (int, long) のテスト
        /// </summary>
        [TestMethod()]
        public void CalcSendPercentageTest02()
        {
            SenderBase<string> target = new SenderBaseImpl();
            TERASOLUNA_Fw_Client_Communication_SenderBaseImplAccessor accessor = new TERASOLUNA_Fw_Client_Communication_SenderBaseImplAccessor(target);
            int actual = accessor.CalcSendPercentage(1, 1);

            Assert.AreEqual(50, actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<>), LogLevel.INFO));
        }

        /// <summary>
        /// CalcSendPercentage (int, long) のテスト
        /// </summary>
        [TestMethod()]
        public void CalcSendPercentageTest03()
        {
            SenderBase<string> target = new SenderBaseImpl();
            TERASOLUNA_Fw_Client_Communication_SenderBaseImplAccessor accessor = new TERASOLUNA_Fw_Client_Communication_SenderBaseImplAccessor(target);
            int actual = accessor.CalcSendPercentage(2, 3);

            Assert.AreEqual(2 * 50 / 3, actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<>), LogLevel.INFO));
        }
    }

    [System.Diagnostics.DebuggerStepThrough()]
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Microsoft.VisualStudio.TestTools.UnitTestGeneration", "1.0.0.0")]
    internal class TERASOLUNA_Fw_Client_Communication_SenderBaseImplAccessor : BaseAccessor
    {
        protected static PrivateType m_privateType = new PrivateType(typeof(SenderBaseImpl));

        internal TERASOLUNA_Fw_Client_Communication_SenderBaseImplAccessor(SenderBase<string> target)
            : base(target, m_privateType)
        {
        }

        internal int CalcSendPercentage(int receiveSize, long totalSize)
        {
            object[] args = new object[] {
                receiveSize,
                totalSize};
            int ret = ((int)(m_privateObject.Invoke("CalcSendPercentage", new System.Type[] {
                    typeof(int),
                    typeof(long)}, args)));
            return ret;
        }
    }
}
