// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.IO;
using TERASOLUNA.Fw.Client.Communication;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Security.AccessControl;
namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.Multipart.MultipartFileElement のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Communication.Multipart.MultipartFileElement 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class MultipartFileElementTest
    {
        private TestContext testContextInstance;

        private static string SIZE_259_FILE_NAME = @"C:\____10________20________30________40________50________60________70________80________90_______100_______110_______120_______130_______140_______150_______160_______170_______180_______190_______200_______210_______220_______230_______240_______250______259";

        private static string SIZE_260_FILE_NAME = @"C:\____10________20________30________40________50________60________70________80________90_______100_______110_______120_______130_______140_______150_______160_______170_______180_______190_______200_______210_______220_______230_______240_______250_______260";

        private static string CANNOT_ACCESS_FILE_NAME = @"C:\NotAccess.txt";

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        [ClassInitialize()]
        public static void MyClassInitialize(TestContext testContext)
        {
            Stream stream = File.Create(SIZE_259_FILE_NAME, 1000, FileOptions.None);
            stream.Close();
        }

        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        [ClassCleanup()]
        public static void MyClassCleanup()
        {
            File.Delete(SIZE_259_FILE_NAME);
        }
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        /// CreateBodyStream (Encoding) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の encoding が null 、プロパティの UploadFilePath が 存在するファイルパスの場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest01()
        {
            // 準備
            Encoding encoding = null;
            string filePath = SIZE_259_FILE_NAME;
            MultipartFileElement target = new MultipartFileElement("TERASOLUNA");
            target.UploadFilePath = filePath;

            // 実施
            FileStream actualStream = target.CreateBodyStream(encoding) as FileStream;

            // 結果
            string expectedFilePath = Path.GetFullPath(filePath);
            Assert.AreEqual(0L, actualStream.Position);
            Assert.IsTrue(actualStream.CanSeek);
            Assert.AreEqual(expectedFilePath, actualStream.Name);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
        }

        /// <summary>
        /// CreateBodyStream (Encoding) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の encoding が not null 、プロパティの UploadFilePath が 存在するファイルパスの場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest02()
        {
            // 準備
            Encoding encoding = Encoding.UTF8;
            string filePath = SIZE_259_FILE_NAME;
            MultipartFileElement target = new MultipartFileElement("TERASOLUNA");
            target.UploadFilePath = filePath;

            // 実施
            FileStream actualStream = target.CreateBodyStream(encoding) as FileStream;

            // 結果
            string expectedFilePath = Path.GetFullPath(filePath);
            Assert.AreEqual(0L, actualStream.Position);
            Assert.IsTrue(actualStream.CanSeek);
            Assert.AreEqual(expectedFilePath, actualStream.Name);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
        }

        /// <summary>
        /// CreateBodyStream (Encoding) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の encoding が not null 、プロパティの UploadFilePath が null の場合、
        /// InvalidOperationException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest03()
        {
            // 準備
            Encoding encoding = Encoding.UTF8;
            string filePath = null;
            MultipartFileElement target = new MultipartFileElement("TERASOLUNA");
            TERASOLUNA_Fw_Client_Communication_MultipartFileElementAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartFileElementAccessor(target);
            accessor._uploadFilePath = filePath;

            // 実施
            try
            {
                FileStream actualStream = target.CreateBodyStream(encoding) as FileStream;
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 確認
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_NO_FILENAME"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartFileElement), typeof(InvalidOperationException), "E_COMMUNICATION_NO_FILENAME"));
        }

        /// <summary>
        /// CreateBodyStream (Encoding) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の encoding が not null 、プロパティの UploadFilePath が存在しないファイルパスの場合、
        /// FileNotFoundException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest04()
        {
            // 準備
            Encoding encoding = Encoding.UTF8;
            string filePath = "NotExist.txt";
            MultipartFileElement target = new MultipartFileElement("TERASOLUNA");
            target.UploadFilePath = filePath;

            // 実施
            try
            {
                FileStream actualStream = target.CreateBodyStream(encoding) as FileStream;
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (FileNotFoundException e)
            {
                // 確認
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_FILE_NOT_FOUND", filePath), e.Message);
                Assert.AreEqual(filePath, e.FileName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartFileElement), typeof(FileNotFoundException), "E_COMMUNICATION_FILE_NOT_FOUND", filePath));
        }

        /// <summary>
        /// CreateBodyStream (Encoding) のテスト
        /// </summary>
        /// <remarks>
        /// 本テストを実行する場合、以下の条件を見たなさなければならない。
        /// <list type="bullet">
        /// <item>
        /// "C:\NotAccess.txt" を作成しておくこと。
        /// </item>
        /// <item>
        /// "C:\NotAccess.txt" ファイルは読み取り禁止にしておくこと。
        /// </item>
        /// </list>
        /// 引数の encoding が not null 、プロパティの UploadFilePath のファイルパスにアクセス権がない場合、
        /// UnauthorizedAccessException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest05()
        {
            // 準備
            Encoding encoding = Encoding.UTF8;
            string filePath = CANNOT_ACCESS_FILE_NAME;
            MultipartFileElement target = new MultipartFileElement("TERASOLUNA");
            target.UploadFilePath = filePath;

            // 実施
            try
            {
                FileStream actualStream = target.CreateBodyStream(encoding) as FileStream;
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (UnauthorizedAccessException)
            {
                // 確認
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
        }

        /// <summary>
        /// CreateBodyStream (Encoding) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の encoding が not null 、プロパティの UploadFilePath のファイル名が260バイトの場合、
        /// FileNotFoundException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest06()
        {
            // 準備
            Encoding encoding = Encoding.UTF8;
            string filePath = SIZE_260_FILE_NAME;
            MultipartFileElement target = new MultipartFileElement("TERASOLUNA");
            target.UploadFilePath = filePath;

            // 実施
            try
            {
                FileStream actualStream = target.CreateBodyStream(encoding) as FileStream;
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (FileNotFoundException e)
            {
                // 確認
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_FILE_NOT_FOUND", filePath), e.Message);
                Assert.AreEqual(filePath, e.FileName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartFileElement), typeof(FileNotFoundException), "E_COMMUNICATION_FILE_NOT_FOUND", filePath));
        }

        /// <summary>
        /// CreateHeader () のテスト
        /// </summary>
        /// <remarks>
        /// プロパティの UploadFilePath が存在するファイルパス、プロパティの Name が not null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateHeaderTest01()
        {
            // 準備
            string name = "TERASOLUNA";
            string filePath = "terasoluna.txt";
            MultipartFileElement target = new MultipartFileElement(name);
            target.UploadFilePath = filePath;

            // 実施
            string actualHeaderString = target.CreateHeader();

            // 結果
            StringBuilder expectedMessage = new StringBuilder();
            expectedMessage.AppendLine(string.Format("Content-Disposition: form-data; name={0}; filename={1}", name, filePath));
            expectedMessage.Append("Content-Type: application/octet-stream");
            Assert.AreEqual(expectedMessage.ToString(), actualHeaderString);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
        }

        /// <summary>
        /// MultipartFileElement (string) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の name が not null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            // 準備
            string name = "terasoluna";

            MultipartFileElement target = new MultipartFileElement(name);

            // 結果
            TERASOLUNA_Fw_Client_Communication_MultipartElementAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartElementAccessor(target);
            Assert.AreEqual(name, accessor._name);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
        }

        /// <summary>
        /// MultipartFileElement (string) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の name が null の場合、
        /// ArgumentNullException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorTest02()
        {
            // 準備
            string name = null;

            try
            {
                MultipartFileElement target = new MultipartFileElement(name);
            }
            catch (ArgumentNullException e)
            {
                // 確認
                Assert.AreEqual("name", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
        }

        /// <summary>
        /// MultipartFileElement (string) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の name が空文字の場合、
        /// ArgumentException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorTest03()
        {
            // 準備
            string name = string.Empty;

            try
            {
                MultipartFileElement target = new MultipartFileElement(name);
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "name"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
        }

        /// <summary>
        /// UploadFilePath のテスト
        /// </summary>
        /// <remarks>
        /// 空文字でない文字列を設定した場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void UploadFilePathTest01()
        {
            // 準備
            string uploadFilePath = "terasoluna.txt";
            MultipartFileElement target = new MultipartFileElement("TERASOLUNA");

            // 実施
            target.UploadFilePath = uploadFilePath;

            // 確認
            TERASOLUNA_Fw_Client_Communication_MultipartFileElementAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartFileElementAccessor(target);
            Assert.AreEqual(uploadFilePath, accessor._uploadFilePath);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
        }

        /// <summary>
        ///UploadFilePath のテスト
        ///</summary>
        /// <remarks>
        /// null を設定した場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void UploadFilePathTest02()
        {
            // 準備
            string uploadFilePath = null;
            MultipartFileElement target = new MultipartFileElement("TERASOLUNA");

            try
            {
                // 実施
                target.UploadFilePath = uploadFilePath;
            }
            catch (ArgumentNullException e)
            {
                Assert.AreEqual("UploadFilePath", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartFileElement), typeof(ArgumentNullException), "E_NULL_PROPERTY_VALUE", "UploadFilePath"));
        }

        /// <summary>
        /// UploadFilePath のテスト
        /// </summary>
        /// <remarks>
        /// 空文字列を設定した場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void UploadFilePathTest03()
        {
            // 準備
            string uploadFilePath = string.Empty;
            MultipartFileElement target = new MultipartFileElement("TERASOLUNA");

            try
            {
                // 実施
                target.UploadFilePath = uploadFilePath;
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_PROPERTY_VALUE", "UploadFilePath"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartFileElement), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartFileElement), typeof(ArgumentException), "E_EMPTY_PROPERTY_VALUE", "UploadFilePath"));
        }
    }
}
