// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.BLogic;
using TERASOLUNA.Fw.Common.BLogic;
using System.Data;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Reflection;
using TERASOLUNA.Fw.ClientUT.BLogic.BinaryFileUploadBLogicData;
using TERASOLUNA.Fw.Client.Communication;

namespace TERASOLUNA.Fw.ClientUT.BLogic
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.BLogic.BinaryFileUploadBLogic&lt;T&gt; のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.BLogic.BinaryFileUploadBLogic&lt;T&gt; 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class BinaryFileUploadBLogicTest
    {


        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion



        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が0の場合、ArgumentExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest01()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            // 引数
            BLogicParam param = new BLogicParam();
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, "TERASOLUNA_REQUEST");

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            param.ParamData = dataSet;

            try
            {
                target.Execute(param);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch(ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_TABLE_NOT_FOUND", param.ParamData.GetType().FullName), e.Message);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<string>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルがない場合、
        /// ArgumentExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest02()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            // 引数
            BLogicParam param = new BLogicParam();
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, "TERASOLUNA_REQUEST");

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("Other");
            dataTable.Columns.Add("FilePath");
            dataTable.Columns.Add("Value");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA01";
            dataRow["Value"] = "TERASOLUNA02";
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            try
            {
                target.Execute(param);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_TABLE_NOT_FOUND", param.ParamData.GetType().FullName), e.Message);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<string>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が0の場合、
        /// ArgumentExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest03()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            // 引数
            BLogicParam param = new BLogicParam();
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, "TERASOLUNA_REQUEST");

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            try
            {
                target.Execute(param);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_RECORD_NOT_FOUND", param.ParamData.GetType().FullName), e.Message);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<string>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }


        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が1、
        /// かつFilePathカラムがない場合、
        /// ArgumentExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest04()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            // 引数
            BLogicParam param = new BLogicParam();
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, "TERASOLUNA_REQUEST");

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            DataRow dataRow = dataTable.NewRow();
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            try
            {
                target.Execute(param);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_COLUMN_NOT_FOUND", param.ParamData.GetType().FullName), e.Message);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<string>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }



        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が1、
        /// かつFilePathカラムの値がnullの場合、
        /// ArgumentExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest05()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            // 引数
            BLogicParam param = new BLogicParam();
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, "TERASOLUNA_REQUEST");

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = DBNull.Value;
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            try
            {
                target.Execute(param);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);

                FieldInfo binaryFileTableName = target.GetType().GetField("BINARY_FILE_TABLE_NAME", BindingFlags.Static | BindingFlags.NonPublic);
                FieldInfo binaryFileColumnName = target.GetType().GetField("BINARY_FILE_COLUMN_NAME", BindingFlags.Static | BindingFlags.NonPublic);

                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_INVALID_DTOTABLE", binaryFileTableName.GetValue(null), binaryFileColumnName.GetValue(null)), e.Message);
            }
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が1、
        /// かつFilePathカラムの値が空文字列の場合、
        /// ArgumentExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest06()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            // 引数
            BLogicParam param = new BLogicParam();
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, "TERASOLUNA_REQUEST");

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = string.Empty;
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            try
            {
                target.Execute(param);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);

                FieldInfo binaryFileTableName = target.GetType().GetField("BINARY_FILE_TABLE_NAME", BindingFlags.Static | BindingFlags.NonPublic);
                FieldInfo binaryFileColumnName = target.GetType().GetField("BINARY_FILE_COLUMN_NAME", BindingFlags.Static | BindingFlags.NonPublic);

                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_INVALID_DTOTABLE", binaryFileTableName.GetValue(null), binaryFileColumnName.GetValue(null)), e.Message);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<string>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が1、
        /// かつFilePathカラムの値が正常値の場合、
        /// 正常に実行されることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest07()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub01 stub = new BinaryFileUploadBLogic_CommunicatorStub01();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            BLogicResult actual = target.Execute(param);

            // 戻り値確認
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);
            Assert.AreEqual(0, actual.Errors.Count);
            Assert.AreSame(stub._resultData, actual.ResultData);

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreEqual(1, stub._requestHeaders.Count);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が2、
        /// かつFilePathカラムの値が正常値、かつFilePathの2行目の値がnullの場合、
        /// 正常に実行されることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest08()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub01 stub = new BinaryFileUploadBLogic_CommunicatorStub01();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataTable.Rows.Add(dataRow);

            // 引数のデータセットの２行目以降
            DataRow dataRow2 = dataTable.NewRow();
            dataRow2["FilePath"] = DBNull.Value;
            dataTable.Rows.Add(dataRow2);

            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            BLogicResult actual = target.Execute(param);

            // 戻り値確認
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);
            Assert.AreEqual(0, actual.Errors.Count);
            Assert.AreSame(stub._resultData, actual.ResultData);

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreEqual(1, stub._requestHeaders.Count);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が2、
        /// かつFilePathカラムの値が正常値、かつFilePathの2行目の値が空文字列の場合、
        /// 正常に実行されることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest09()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub01 stub = new BinaryFileUploadBLogic_CommunicatorStub01();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataTable.Rows.Add(dataRow);

            // 引数のデータセットの２行目以降
            DataRow dataRow2 = dataTable.NewRow();
            dataRow2["FilePath"] = string.Empty;
            dataTable.Rows.Add(dataRow2);

            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            BLogicResult actual = target.Execute(param);

            // 戻り値確認
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);
            Assert.AreEqual(0, actual.Errors.Count);
            Assert.AreSame(stub._resultData, actual.ResultData);

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreEqual(1, stub._requestHeaders.Count);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
        }


        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が2、
        /// かつFilePathカラムの値が正常値、かつFilePathの2行目の値が正常値の場合、
        /// 正常に実行されることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest10()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub01 stub = new BinaryFileUploadBLogic_CommunicatorStub01();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataTable.Rows.Add(dataRow);

            // 引数のデータセットの２行目以降
            DataRow dataRow2 = dataTable.NewRow();
            dataRow2["FilePath"] = string.Empty;
            dataTable.Rows.Add(dataRow2);

            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            BLogicResult actual = target.Execute(param);

            // 戻り値確認
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);
            Assert.AreEqual(0, actual.Errors.Count);
            Assert.AreSame(stub._resultData, actual.ResultData);

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreEqual(1, stub._requestHeaders.Count);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が1、
        /// かつFilePathカラムの値が正常値、かつValueカラムが存在する場合、
        /// 正常に実行されることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest11()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub01 stub = new BinaryFileUploadBLogic_CommunicatorStub01();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット（FilePath以外のValueカラム含む）
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            dataTable.Columns.Add("Value");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataRow["Value"] = "Other";
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            BLogicResult actual = target.Execute(param);

            // 戻り値確認
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);
            Assert.AreEqual(0, actual.Errors.Count);
            Assert.AreSame(stub._resultData, actual.ResultData);

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreEqual(1, stub._requestHeaders.Count);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
        }


        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 引数のデータセットのテーブル数が1、かつFileテーブルの行数が2、
        /// かつFilePathカラムの値が正常値、かつFileテーブル以外のテーブルが存在する場合、
        /// 正常に実行されることの確認。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest12()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub01 stub = new BinaryFileUploadBLogic_CommunicatorStub01();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataTable.Rows.Add(dataRow);

            // 引数のFileテーブル以外のテーブル１
            DataTable dataTable2 = new DataTable("Other");
            dataTable2.Columns.Add("FilePath");
            DataRow dataRow2 = dataTable2.NewRow();
            dataRow2["FilePath"] = "OTHER";
            dataTable2.Rows.Add(dataRow2);

            // 引数のFileテーブル以外のテーブル２
            DataTable dataTable3 = new DataTable("Other2");
            dataTable3.Columns.Add("FilePath2");
            DataRow dataRow3 = dataTable3.NewRow();
            dataRow3["FilePath2"] = "OTHER2";
            dataTable3.Rows.Add(dataRow3);

            dataSet.Tables.Add(dataTable);
            dataSet.Tables.Add(dataTable2);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            BLogicResult actual = target.Execute(param);

            // 戻り値確認
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);
            Assert.AreEqual(0, actual.Errors.Count);
            Assert.AreSame(stub._resultData, actual.ResultData);

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreEqual(1, stub._requestHeaders.Count);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 処理実行時にArgumentExceptionがスローされた場合、
        /// ログが出力され、そのまま例外がスローされることを確認する。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest13()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub02 stub = new BinaryFileUploadBLogic_CommunicatorStub02();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            try
            {
                BLogicResult actual = target.Execute(param);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 成功
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual("ArgumentExceptionTest", e.Message);
            }

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<string>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 処理実行時にServerException（例外種別：serverException）がスローされた場合、
        /// ログが出力され、そのまま例外がスローされることを確認する。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest14()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub03 stub = new BinaryFileUploadBLogic_CommunicatorStub03();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            try
            {
                BLogicResult actual = target.Execute(param);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                // 成功
                Assert.AreEqual("serverException", e.ErrorType);
                Assert.AreEqual("ServerExceptionTest", e.Message);
            }

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<string>), typeof(ServerException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 処理実行時にServerException（例外種別：serverValidateException）がスローされた場合、
        /// ログが出力され、ServerExceptionのErrorsの内容が戻り値に正しく格納されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest15()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub04 stub = new BinaryFileUploadBLogic_CommunicatorStub04();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            BLogicResult actual = target.Execute(param);

            // 戻り値確認
            Assert.AreEqual("serverValidateException", actual.ResultString);
            Assert.AreEqual(1, actual.Errors.Count);
            Assert.AreSame(stub._list[0], actual.Errors[0]);

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreEqual(1, stub._requestHeaders.Count);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<string>), typeof(ServerException), "W_BLOGIC_EXCEPTION"));
        }


        /// <summary>
        ///Communicate (DataSet, IDictionary&lt;string,string&gt;) のテスト
        ///</summary>
        ///<remarks>
        /// 処理実行時にCommunicationExceptionがスローされた場合、
        /// ログが出力され、CommunicationExceptionのメッセージの内容が戻り値に正しく格納されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void CommunicateTest16()
        {
            BinaryFileUploadBLogic<DataSet> target = new BinaryFileUploadBLogic<DataSet>();

            BinaryFileUploadBLogic_CommunicatorStub05 stub = new BinaryFileUploadBLogic_CommunicatorStub05();

            typeof(CommunicateBLogicBase<string>).GetField("_communicator", BindingFlags.NonPublic | BindingFlags.Instance).SetValue(target, stub);

            // 引数
            BLogicParam param = new BLogicParam();
            string requestName = "TERASOLUNA_REQUEST";
            param.Items.Add(BinaryFileUploadBLogic<DataSet>.REQUEST_NAME, requestName);

            // 引数のデータセット
            DataSet dataSet = new DataSet();
            DataTable dataTable = new DataTable("File");
            dataTable.Columns.Add("FilePath");
            DataRow dataRow = dataTable.NewRow();
            dataRow["FilePath"] = "TERASOLUNA";
            dataTable.Rows.Add(dataRow);
            dataSet.Tables.Add(dataTable);
            dataSet.AcceptChanges();

            param.ParamData = dataSet;

            BLogicResult actual = target.Execute(param);

            // 戻り値確認
            Assert.AreEqual(BinaryFileUploadBLogic<DataSet>.COMMUNICATION_EXCEPTION, actual.ResultString);

            Assert.AreEqual(1, actual.Errors.Count);
            Assert.AreEqual(BinaryFileUploadBLogic<DataSet>.COMMUNICATION_EXCEPTION, actual.Errors[0].Key);
            Assert.AreSame(stub._exceptionMessage, actual.Errors[0].Message);

            // Communicateメソッドの呼び出し確認
            Assert.AreEqual("TERASOLUNA", stub._filePath);
            Assert.AreEqual(1, stub._requestHeaders.Count);
            Assert.AreSame(requestName, stub._requestHeaders[BinaryFileUploadBLogic<DataSet>.REQUEST_NAME]);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<string>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<string>), typeof(CommunicationException), "W_BLOGIC_EXCEPTION"));
        }

    }


}
