﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Configuration;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Configuration.Conversion
{
    /// <summary>
    /// データセット変換設定ファイル内の convert 要素のコレクションを格納する構成要素を表します。
    /// </summary>
    /// <remarks> 
    /// <see cref="ConvertCollection"/> は、conversionConfiguration 要素の convert 要素のコレクションを表します。 
    /// </remarks>
    public class ConvertCollection : ConfigurationElementCollection
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(ConvertCollection));

        /// <summary>
        /// convert 要素の要素名です。
        /// </summary>
        /// <remarks>
        /// この定数の値は "convert" です。
        /// </remarks>
        private static readonly string CONVERT_NAME = "convert";

        /// <summary>
        /// <see cref="ConvertCollection"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// デフォルトコンストラクタです。
        /// </remarks>
        public ConvertCollection()
        {
        }

        /// <summary>
        /// 新しい <see cref="ConvertElement"/> を作成します。
        /// </summary>
        /// <returns>新しい <see cref="ConvertElement"/> 。</returns>
        protected override ConfigurationElement CreateNewElement()
        {
            return new ConvertElement();
        }

        /// <summary>
        /// 指定した convert 要素の要素キーを取得します。
        /// </summary>
        /// <remarks>
        /// string 型を返します。
        /// </remarks>
        /// <param name="element">キーを返す <see cref="ConvertElement"/> 。</param>
        /// <returns>指定した <see cref="ConvertElement"/> のキーとなる id 属性の値。</returns>
        /// <exception cref="System.ArgumentNullException">element が null 参照です。</exception>
        /// <exception cref="System.ArgumentException">element が <see cref="ConvertElement"/> ではありません。</exception>
        protected override object GetElementKey(ConfigurationElement element)
        {
            if (element == null)
            {
                ArgumentNullException exception = new ArgumentNullException("element");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Client.Properties.Resources.E_NULL_ARGUMENT, "element"), exception);
                }
                throw exception;
            }

            ConvertElement ConvertElement = element as ConvertElement;
            if (ConvertElement == null)
            {
                string message = string.Format(Client.Properties.Resources.E_INVALID_CAST, element.GetType().FullName, typeof(ConvertElement).FullName);
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            return ConvertElement.Id;
        }

        /// <summary>
        /// <see cref="ConvertCollection"/> に convert 要素を追加します。
        /// </summary>
        /// <remarks>
        /// <see cref="ConfigurationElementCollection.BaseAdd(ConfigurationElement)"/> メソッドを使用して、
        /// コレクションに新しい <see cref="ConvertElement"/> を追加します。
        /// </remarks>
        /// <param name="element">追加する <see cref="ConvertElement"/> 。</param>
        public virtual void Add(ConvertElement element)
        {
            BaseAdd(element);
        }
        
        /// <summary>
        /// <see cref="ConvertCollection"/> が読み取り専用かどうかを示す値を取得します。 
        /// </summary>
        /// <returns>
        /// <see cref="ConvertCollection"/> が読み取り専用の場合は true 。それ以外の場合は false 。
        /// </returns>
        /// <remarks>
        /// <see cref="ConvertCollection"/> は読み取り専用ではありません。
        /// </remarks>
        public override bool IsReadOnly()
        {
            return false;
        }

        /// <summary>
        /// 指定したインデックス位置にある convert 要素を取得します。
        /// </summary>
        /// <remarks>
        /// <see cref="ConfigurationElementCollection.BaseGet(int)"/> メソッドを使用して、
        /// <see cref="ConvertCollection"/> から <see cref="ConvertElement"/> を取得します。
        /// </remarks>
        /// <param name="index">取得する <see cref="ConvertElement"/> のインデックス位置。</param>
        /// <value>
        /// 指定したインデックス位置にある <see cref="ConvertElement"/> 。
        /// </value>
        public ConvertElement this[int index]
        {
            get
            {
                return (ConvertElement)BaseGet(index);
            }
        }

        /// <summary>
        /// 指定した id を持つ convert 要素を取得します。
        /// </summary>
        /// <param name="id">取得する convert 要素の id 。</param>
        /// <value>指定した id を持つ <see cref="ConvertElement"/> 。それ以外の場合は null 参照。</value>
        /// <remarks>
        /// <see cref="ConfigurationElementCollection.BaseGet(object)"/> メソッドを使用して、
        /// <see cref="ConvertCollection"/> から <see cref="ConvertElement"/> を取得します。
        /// <para>
        /// 指定したキーを持つ <see cref="ConvertElement"/> がコレクションに存在しない場合、null 参照を返します。
        /// </para>
        /// </remarks>
        public new ConvertElement this[string id]
        {
            get
            {
                return (ConvertElement)BaseGet(id);
            }
        }

        /// <summary>
        /// データセット変換設定ファイル内のこの要素のコレクションを識別するために使用する名前を取得します。
        /// </summary>
        /// <value>
        /// この要素のコレクションの要素名。
        /// </value>
        protected override string ElementName
        {
            get
            {
                return CONVERT_NAME;
            }
        }
    }
}
