﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System.Configuration;
using System.Data;

namespace TERASOLUNA.Fw.Client.Configuration.Conversion
{
    /// <summary>
    /// データセット変換設定ファイル内の column 要素を表します。
    /// </summary>
    /// <remarks>
    /// 新規 <see cref="DataSet"/> への変換、または値を持つ既存の <see cref="DataSet"/> への変換情報
    /// を定義する要素です。 column 要素は以下の属性を含みます。
    /// また、<see cref="TERASOLUNA.Fw.Client.Conversion.DataSetConverter"/>
    /// を利用する場合は、各属性は以下のような制限に基づいて記述する必要があります。
    /// <list type="table">
    /// <listheader>
    /// <description>属性</description>
    /// <description>説明</description>
    /// </listheader>
    /// <item>
    /// <description>src</description>
    /// <description>
    /// 必須の属性です。<see cref="DataSet"/> のテーブル名とカラム名を、"." 区切りで指定します。
    /// </description>
    /// </item>
    /// <item>
    /// <description>dest</description>
    /// <description>
    /// 省略可能な属性です。(<see cref="DataSet"/>) のテーブル名とカラム名を、"." 区切りで指定します。
    /// 省略した場合、src 属性の値と同じになります。
    /// </description>
    /// </item>
    /// </list>
    /// </remarks>
    public sealed class ColumnElement : ConfigurationElement
    {
        /// <summary>
        /// src 属性を表す名前です。
        /// </summary>
        /// <remarks>
        /// この定数の値は "src" です。
        /// </remarks>
        private const string ATTRIBUTE_NAME_SRC_NAME = "src";

        /// <summary>
        /// dest 属性を表す名前です。
        /// </summary>
        /// <remarks>
        /// この定数の値は "dest" です。
        /// </remarks>
        private const string ATTRIBUTE_NAME_DEST_NAME = "dest";

        /// <summary>
        /// <see cref="ColumnElement"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// デフォルトコンストラクタです。
        /// </remarks>
        public ColumnElement()
        {
        }

        /// <summary>
        /// <see cref="ColumnElement"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// column 要素の属性の値を初期化します。
        /// </remarks>
        /// <param name="src">src 属性に指定されたカラム名。</param>
        /// <param name="dest">dest 属性に指定されたカラム名。</param>
        public ColumnElement(string src, string dest)
        {
            this[ATTRIBUTE_NAME_SRC_NAME] = src;
            this[ATTRIBUTE_NAME_DEST_NAME] = dest;
        }

        /// <summary>
        /// src 属性の値を取得します。
        /// </summary>
        /// <value>
        /// src 属性の値。
        /// </value>
        [ConfigurationProperty(ATTRIBUTE_NAME_SRC_NAME, IsRequired = true, IsKey = true)]
        public string SrcName
        {
            get
            {
                return (string)this[ATTRIBUTE_NAME_SRC_NAME];
            }
        }

        /// <summary>
        /// dest 属性の値を取得します。
        /// </summary>
        /// <value>
        /// dest 属性の値。
        /// </value>
        [ConfigurationProperty(ATTRIBUTE_NAME_DEST_NAME)]
        public string DestName
        {
            get
            {
                return (string)this[ATTRIBUTE_NAME_DEST_NAME];
            }
        }
    }
}
