﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.IO;
using System.Net;
using System.Text;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Communication
{
    /// <summary>
    /// マルチパートデータを格納するデータ型の基底クラスです。
    /// </summary>
    /// <remarks>
    /// <para>マルチパートデータを格納するためのデータ型は、 <see cref="MultipartElement"/> の派生クラスとして
    /// 作成します。</para>
    /// <para>派生クラスでは、文字列引数を一つとるコンストラクタを作成する必要があります。</para>
    /// <para>このクラスは抽象クラスです。</para>
    /// </remarks>
    [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Design", "CA1012:AbstractTypesShouldNotHaveConstructors")]
    public abstract class MultipartElement
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(MultipartElement));

        /// <summary>
        /// マルチパートデータのヘッダに Content-Disposition ヘッダを設定する際、キーとして利用する文字列です。
        /// </summary>
        /// <remarks>
        /// <para>定数の値は "Content-Disposition" です。</para>
        /// </remarks>
        protected static readonly string CONTENT_DISPOSITION = "Content-Disposition";

        /// <summary>
        /// マルチパートデータのヘッダに Content-Type ヘッダを設定する際、キーとして利用する文字列です。
        /// </summary>
        /// <remarks>
        /// <para>定数の値は "Content-Type" です。</para>
        /// </remarks>
        protected static readonly string CONTENT_TYPE = "Content-Type";

        /// <summary>
        /// HTTP ヘッダを出力する際に利用する文字列です。
        /// </summary>
        /// <remarks>
        /// <para>この定数の値は "{0}: {1}" です。</para>
        /// </remarks>
        protected static readonly string HEADER_FORMAT_STRING = "{0}: {1}";

        /// <summary>
        /// 改行文字として利用する文字列です。
        /// </summary>
        /// <remarks>
        /// <para>この定数の値は "\r\n" です。</para>
        /// </remarks>
        protected static readonly string NEWLINE_STRING = "\r\n";

        /// <summary>
        /// マルチパートの要素名です。
        /// </summary>
        private string _name = null;

        /// <summary>
        /// <see cref="MultipartElement"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// <para><paramref name="name"/> を引数とするコンストラクタです。</para>
        /// <para>派生クラスを作成する際、必ずコンストラクタで <paramref name="name"/> を設定する必要があります。</para>
        /// </remarks>
        /// <param name="name">マルチパートの要素名です。</param>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="name"/> が <c>null</c> 参照です。
        /// </exception>
        /// <exception cref="ArgumentException">
        /// <paramref name="name"/> が空文字です。
        /// </exception>
        public MultipartElement(string name)
        {
            if (name == null)
            {
                ArgumentNullException exception = new ArgumentNullException("name");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "name"), exception);
                }
                throw exception;
            }
            if (name.Length == 0)
            {
                string message = string.Format(Properties.Resources.E_EMPTY_STRING, "name");
                ArgumentException exception = new ArgumentException(message);
                if(_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            _name = name;
        }

        /// <summary>
        /// マルチパートの要素名を取得します。
        /// </summary>
        /// <value>
        /// マルチパートの要素名です。
        /// </value>
        public string Name
        {
            get
            {
                return _name;
            }
        }

        /// <summary>
        /// マルチパートデータのボディを読み出すための <see cref="Stream"/> を返却します。
        /// </summary>
   　　 /// <param name="encoding">文字列を本文とするマルチパートデータをエンコードするための <see cref="Encoding"/>。</param>
        /// <remarks>        
        /// <para>マルチパートデータのボディとなる<see cref="Stream"/>を返却します。</para>
        /// <para>抽象メソッドです。</para>
        /// <para>派生クラスで実装します。</para>
        /// <para>実装する場合には、返却する <see cref="Stream"/> のPositionが "0" であることを保証する必要があります。
        /// また、返却する <see cref="Stream"/> はシーク可能(CanSeekが "true" を返す)であることを保証する必要があります。</para>
        /// </remarks>
        /// <returns>マルチパートデータのボディを読み出すための <see cref="Stream"/>。</returns>
        public abstract Stream CreateBodyStream(Encoding encoding);

        /// <summary>
        /// マルチパートデータのヘッダとなる文字列を返却します。
        /// </summary>
        /// <remarks>
        /// <para>マルチパートデータのヘッダとなる文字列を返却します。</para>
        /// <para>抽象メソッドです。</para>
        /// <para>派生クラスで実装します。</para>
        /// <para>マルチパートデータのヘッダは、以下のフォーマットを想定しています。</para>
        /// <para>Content-Disposition : form-data; name= マルチパートの要素名 (ファイルの場合は ;filename= フルパスのファイル名)</para>
        /// <para>Content-Type : 送信するデータパターン毎に決められた値</para>
        /// <para>拡張時に必要に応じて、ヘッダ情報を追加してください。</para>
        /// <para>使用できる文字は、 <see cref="HttpWebRequest.Headers"/> に利用可能な文字となります。</para>
        /// </remarks>
        /// <returns>マルチパートデータのヘッダとなる文字列です。</returns>
        public abstract string CreateHeader();
    }
}
