/**
 *   Copyright 2007 Y.Murakamin
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package net.murakamin.sticker.commands;

import net.murakamin.sticker.CommandRunner;
import net.murakamin.sticker.StickerContext;
import net.murakamin.sticker.commands.exception.CommandExecutionException;
import net.murakamin.sticker.commands.util.StickerContextUtil;

import org.apache.commons.lang.SystemUtils;

/**
 * Command that shows cursor definition
 * 
 * @author Y.Murakamin
 * @see net.murakamin.sticker.commands.CursorsCommand
 * @see net.murakamin.sticker.commands.ForCommand
 * @see net.murakamin.sticker.commands.IfCommand
 * @since 0.1.3
 */
public class CursorCommand implements Command
{

	/**
	 * Key name prefix to store cursor name in
	 * {@link net.murakamin.sticker.StickerContext}
	 * 
	 * @see net.murakamin.sticker.StickerContext
	 */
	public static final String CURSOR_NAME_PREFIX = Messages
	        .getString("net.murakamin.sticker.commands.CursorCommand.cursor_name_prefix"); //$NON-NLS-1$

	private String name;
	private String sql;
	private StickerContext context;

	/**
	 * The specified name is made a key and cursor information is set to
	 * StickerContext.
	 * 
	 * @see net.murakamin.sticker.commands.ExecutableCommand#execute(CommandRunner)
	 * @throws
	 *      net.murakamin.sticker.commands.exception.CommandExecutionException
	 *      When the same cursor name is specified
	 */
	public void execute(final CommandRunner runner) throws Exception
	{
		this.context = runner.getStickerContext();

		if (context.containsKey(CursorCommand.CURSOR_NAME_PREFIX
		        + this.getName()))
		{
			throw new CommandExecutionException(
			        Messages
			                .getString("net.murakamin.sticker.commands.CursorCommand.Duplicated_cursor_name") //$NON-NLS-1$
			                + this.getName());
		}

		context.put(CursorCommand.CURSOR_NAME_PREFIX + this.getName(), this);
		runner.run(this);
	}

	/**
	 * get the cursor name
	 * 
	 * @return the cursor name
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * get the SQL statement of this cursor
	 * 
	 * @return the SQL statement of this cursor
	 */
	public String getSql()
	{
		return StickerContextUtil.applyContextVariable(this.context, sql);
	}

	/**
	 * get the version term object
	 * 
	 * @return {@link net.murakamin.sticker.commands.VersionTerm}object
	 */
	public VersionTerm getVersionTerm()
	{
		return new VersionTerm("0.1.3"); //$NON-NLS-1$
	}

	/**
	 * The log is output in the debugging log.
	 * 
	 * @return always true
	 * @see net.murakamin.sticker.commands.ExecutableCommand#isDebugPrint()
	 */
	public boolean isDebugPrint()
	{
		return true;
	}

	/**
	 * set the SQL statement of this cursor
	 * 
	 * @param name
	 *            cursor name
	 */
	public void setName(final String name)
	{
		this.name = name;
	}

	/**
	 * set the SQL statement of this cursor
	 * 
	 * @param sql
	 *            the SQL statement of this cursor
	 */
	public void setSql(final String sql)
	{
		this.sql = sql;
	}

	@Override
	public String toString()
	{
		StringBuffer buffer = new StringBuffer();

		buffer.append("<cursor> name : "); //$NON-NLS-1$
		buffer.append(this.getName());
		buffer.append(SystemUtils.LINE_SEPARATOR);
		buffer.append(this.getSql());

		return buffer.toString();
	}

}
