/**
 *   Copyright 2007 Y.Murakamin
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package net.murakamin.sticker.commands;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.SQLException;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import net.murakamin.sticker.CommandRunner;
import net.murakamin.sticker.Commands;
import net.murakamin.sticker.StickerContext;
import net.murakamin.sticker.commands.enums.PostChildCommandExecutionType;
import net.murakamin.sticker.commands.enums.PreChildCommandExecutionType;
import net.murakamin.sticker.commands.exception.CommandExecutionException;
import net.murakamin.sticker.commands.util.StickerContextUtil;

/**
 * A current connection to a remote database server is set. The command of the
 * child element uses the connection set here.
 * 
 * @author Y.Murakamin
 * 
 */
public class ConnectCommand implements ParentCommand,
		ChildCommandExceptionHandler {

	Commands childlen = new Commands();
	private String name;

	/**
	 * @see net.murakamin.sticker.commands.ParentCommand#addChild(ExecutableCommand)
	 */
	public void addChild(final ExecutableCommand child) {
		this.childlen.add(child);
	}

	/**
	 * @see net.murakamin.sticker.commands.ExecutableCommand#execute(CommandRunner)
	 */
	public void execute(final CommandRunner runner) throws Exception {
		try {

			StickerContext context = runner.getStickerContext();
			DatabaseConnectionProperties props = (DatabaseConnectionProperties) context
					.get(ServerCommand.CONTEXT_REMOTE_CONNECTION_PROP);

			Connection con = this.createConnection(context, props);
			con.setAutoCommit(props.getAutocommit());

			runner.getConnectionPool()
					.setRemoteConnection(props.getName(), con);
			runner.run(this);

		} catch (Exception e) {
			throw new CommandExecutionException(this, e);
		}

	}

	/**
	 * @see net.murakamin.sticker.commands.ParentCommand#getChildlen()
	 */
	public Commands getChildlen() {
		return this.childlen;
	}

	/**
	 * get the connection name
	 * 
	 * @return connection name
	 */
	public String getName() {
		return name;
	}

	/**
	 * get the version term object
	 * 
	 * @return {@link net.murakamin.sticker.commands.VersionTerm}object
	 */
	public VersionTerm getVersionTerm() {
		return VersionTerm.PERMANENT_COMMAND;
	}

	/**
	 * The rollback of a remote data base is done.
	 * 
	 * @see net.murakamin.sticker.commands.ChildCommandExceptionHandler#handleException(CommandRunner)
	 */
	public void handleException(final CommandRunner runner) throws Exception {

		DatabaseConnectionProperties props = (DatabaseConnectionProperties) runner
				.getStickerContext().get(
						ServerCommand.CONTEXT_REMOTE_CONNECTION_PROP);
		try{
			if (props.getAutocommit() == false) {
				runner.getConnectionPool().getRemoteConnection().rollback();
			}
		}catch (SQLException e)
		{
			// rollback failure.@do nothing?
		}
	}

	/**
	 * The log is output in the debugging log.
	 * 
	 * @return always true
	 * @see net.murakamin.sticker.commands.ExecutableCommand#isDebugPrint()
	 */
	public boolean isDebugPrint() {
		return true;
	}

	/**
	 * current connection is set to null.
	 * 
	 * @return {@link net.murakamin.sticker.commands.enums.PostChildCommandExecutionType.NOT_REPEAT
	 *         PostChildCommandExecutionType.NOT_REPEAT}
	 * @see net.murakamin.sticker.commands.ParentCommand#postChildCommandExecute(CommandRunner)
	 */
	public PostChildCommandExecutionType postChildCommandExecute(
			final CommandRunner runner) throws Exception {

		DatabaseConnectionProperties props = (DatabaseConnectionProperties) runner
				.getStickerContext().get(
						ServerCommand.CONTEXT_REMOTE_CONNECTION_PROP);
		
		try{
			if (props.getAutocommit() == false) {
				runner.getConnectionPool().getRemoteConnection().rollback();
			}
		}catch (SQLException e)
		{
			// rollback failure. do nothing?
		}
		runner.getConnectionPool().setCurrentConnectionName(null);
		return PostChildCommandExecutionType.NOT_REPEAT;
	}

	/**
	 * set the current connection name
	 * 
	 * @return {@link net.murakamin.sticker.commands.enums.PreChildCommandExecutionType.CHILD_EXECUTE
	 *         PreChildCommandExecutionType.CHILD_EXECUTE}
	 * @see net.murakamin.sticker.commands.ParentCommand#preChildCommandExecute(CommandRunner)
	 */
	public PreChildCommandExecutionType preChildCommandExecute(
			final CommandRunner runner) throws Exception {
		runner.getConnectionPool().setCurrentConnectionName(this.getName());
		return PreChildCommandExecutionType.CHILD_EXECUTE;
	}

	/**
	 * set the connection name
	 * 
	 * @param name
	 */
	public void setName(final String name) {
		this.name = name;
	}

	/**
	 * The connection object to connect it with the server is generated. The jar
	 * file specified that driverJarFile attribute is specified is loaded from
	 * the class path.
	 * 
	 * @param context
	 *            the {@link net.murakamin.sticker.StickerContext
	 *            StickerContext} Object
	 * @param props
	 *            database connection properties
	 * @return database connection object
	 * @throws IOException
	 * @throws ClassNotFoundException
	 * @throws InstantiationException
	 * @throws IllegalAccessException
	 * @throws SQLException
	 */
	@SuppressWarnings("unchecked")
	private Connection createConnection(final StickerContext context,
			final DatabaseConnectionProperties props) throws IOException,
			ClassNotFoundException, InstantiationException,
			IllegalAccessException, SQLException {

		ClassLoader loader = this.getClass().getClassLoader();
		if (props.getDriverJarFile() != null) {
			String driverJarName = StickerContextUtil.applyContextVariable(
					context, props.getDriverJarFile());

			File driverFileName = new File(driverJarName).getCanonicalFile();
			URL driverURL = driverFileName.toURI().toURL();

			loader = URLClassLoader.newInstance(new URL[] { driverURL }, this
					.getClass().getClassLoader());
		}

		String driverName = StickerContextUtil.applyContextVariable(context,
				props.getDriver());
		String user = StickerContextUtil.applyContextVariable(context, props
				.getUser());
		String pass = StickerContextUtil.applyContextVariable(context, props
				.getPass());

		Class<Driver> driverClass = (Class<Driver>) Class.forName(driverName,
				true, loader);

		Driver driverInstance = driverClass.newInstance();

		Properties info = new Properties();
		info.put("user", user);
		info.put("password", pass);

		return driverInstance.connect(props.getUrl(), info);
	}

	@Override
	public String toString() {

		StringBuffer buffer = new StringBuffer();

		buffer.append("<connect> connect to ");
		buffer.append(this.getName());
		buffer.append('.');

		return buffer.toString();
	}
}
