/**
 *   Copyright 2007 Y.Murakamin
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package net.murakamin.sticker.commands;

import java.sql.Connection;

import net.murakamin.sticker.CommandRunner;
import net.murakamin.sticker.Commands;
import net.murakamin.sticker.commands.enums.PostChildCommandExecutionType;
import net.murakamin.sticker.commands.enums.PreChildCommandExecutionType;

/**
 * A current connection to a remote database server is set. The command of the
 * child element uses the connection set here.
 * 
 * @author Y.Murakamin
 * 
 */
public class ConnectCommand implements ParentCommand,
        ChildCommandExceptionHandler
{

	Commands childlen = new Commands();
	private String name;

	/**
	 * @see net.murakamin.sticker.commands.ParentCommand#addChild(ExecutableCommand)
	 */
	public void addChild(final ExecutableCommand child)
	{
		this.childlen.add(child);
	}

	/**
	 * @see net.murakamin.sticker.commands.ExecutableCommand#execute(CommandRunner)
	 */
	public void execute(final CommandRunner runner) throws Exception
	{
		runner.run(this);
	}

	/**
	 * @see net.murakamin.sticker.commands.ParentCommand#getChildlen()
	 */
	public Commands getChildlen()
	{
		return this.childlen;
	}

	/**
	 * get the connection name
	 * 
	 * @return connection name
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * get the version term object
	 * 
	 * @return {@link net.murakamin.sticker.commands.VersionTerm}object
	 */
	public VersionTerm getVersionTerm()
	{
		return VersionTerm.PERMANENT_COMMAND;
	}

	/**
	 * The rollback of a remote data base is done.
	 * 
	 * @see net.murakamin.sticker.commands.ChildCommandExceptionHandler#handleException(CommandRunner)
	 */
	public void handleException(final CommandRunner runner) throws Exception
	{
		Connection currentRemoteConnection = runner.getConnectionPool()
		        .getRemoteConnection();
		if (currentRemoteConnection.getAutoCommit() == false)
		{
			currentRemoteConnection.rollback();
		}
	}

	/**
	 * The log is output in the debugging log.
	 * 
	 * @return always true
	 * @see net.murakamin.sticker.commands.ExecutableCommand#isDebugPrint()
	 */
	public boolean isDebugPrint()
	{
		return true;
	}

	/**
	 * current connection is set to null.
	 * 
	 * @return {@link net.murakamin.sticker.commands.enums.PostChildCommandExecutionType.NOT_REPEAT PostChildCommandExecutionType.NOT_REPEAT}
	 * @see net.murakamin.sticker.commands.ParentCommand#postChildCommandExecute(CommandRunner)
	 */
	public PostChildCommandExecutionType postChildCommandExecute(
	        final CommandRunner runner) throws Exception
	{
		runner.getConnectionPool().setCurrentConnectionName(null);
		return PostChildCommandExecutionType.NOT_REPEAT;
	}

	/**
	 * set the current connection name
	 * 
	 * @return {@link net.murakamin.sticker.commands.enums.PreChildCommandExecutionType.CHILD_EXECUTE PreChildCommandExecutionType.CHILD_EXECUTE}
	 * @see net.murakamin.sticker.commands.ParentCommand#preChildCommandExecute(CommandRunner)
	 */
	public PreChildCommandExecutionType preChildCommandExecute(
	        final CommandRunner runner) throws Exception
	{
		runner.getConnectionPool().setCurrentConnectionName(this.getName());
		return PreChildCommandExecutionType.CHILD_EXECUTE;
	}

	/**
	 * set the connection name
	 * 
	 * @param name
	 */
	public void setName(final String name)
	{
		this.name = name;
	}

	@Override
	public String toString()
	{

		StringBuffer buffer = new StringBuffer();

		buffer.append("<connect> connect to ");
		buffer.append(this.getName());
		buffer.append('.');

		return buffer.toString();
	}
}
