/*
SRSearchEngineIconDatabase.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRSearchEngineIconDatabase.h"

#import "SRConstants.h"

@implementation SRSearchEngineIconDatabase

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (id)sharedInstance
{
    static SRSearchEngineIconDatabase*  _sharedInstance = nil;
    if (!_sharedInstance) {
        _sharedInstance = [[SRSearchEngineIconDatabase alloc] init];
    }
    
    return _sharedInstance;
}

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _iconDict = [[NSMutableDictionary dictionary] retain];
    
    return self;
}

//--------------------------------------------------------------//
#pragma mark -- Controller --
//--------------------------------------------------------------//

- (id<SRAppController>)appController
{
    return _appController;
}

- (void)setAppController:(id<SRAppController>)appController
{
    _appController = appController;
}

//--------------------------------------------------------------//
#pragma mark -- Icon management --
//--------------------------------------------------------------//

+ (NSString*)urlStringForIconWithSearchEngine:(SRSearchEngine*)engine
{
    // Get URL
    NSString*   urlString;
    NSURL*      url;
    urlString = [engine valueForKey:@"templateURL"];
    if (!urlString) {
        return nil;
    }
    url = [NSURL URLWithString:[NSString stringWithFormat:urlString, @"Shiira"]];
    if (!url) {
        return nil;
    }
    
    return [NSString stringWithFormat:@"%@://%@/", [url scheme], [url host]];
}

- (NSImage*)iconForURLString:(NSString*)urlString
{
    return [_iconDict objectForKey:urlString];
}

- (void)setIcon:(NSImage*)icon forURLStirng:(NSString*)urlString
{
    [_iconDict setObject:icon forKey:urlString];
}

//--------------------------------------------------------------//
#pragma mark -- Persistence --
//--------------------------------------------------------------//

- (NSString*)_searchEnginIconPath
{
    // Create path ~/Library/Application Support/Shiira/SearchEngineIcon.dat
    NSString*	searchEngineIconPath;
    searchEngineIconPath = [HMApplicationSupportFolder(@"Shiira") 
            stringByAppendingPathComponent:SRSearchEngineIconFileName];
    return searchEngineIconPath;
}

- (void)loadDatabase
{
    // Get path
    NSString*   path;
    path = [self _searchEnginIconPath];
    if (!path) {
        return;
    }
    
    // Check existence
    if (![[NSFileManager defaultManager] fileExistsAtPath:path]) {
        return;
    }
    
    // Load file
    NSDictionary*   dict;
    dict = [NSDictionary dictionaryWithContentsOfFile:path];
    
    // Convert data to image
    NSEnumerator*   enumerator;
    NSString*       urlString;
    enumerator = [dict keyEnumerator];
    while (urlString = [enumerator nextObject]) {
        NSData*     data;
        NSImage*    image;
        data = [dict objectForKey:urlString];
        image = [[NSImage alloc] initWithData:data];
        if (!image) {
            continue;
        }
        
        [_iconDict setObject:image forKey:urlString];
        [image release];
    }
    
    // Check icon
    NSManagedObjectContext* context;
    NSFetchRequest*         request;
    NSArray*                engines;
    context = [_appController managedObjectContext];
    request = [[NSFetchRequest alloc] init];
    [request setEntity:
            [NSEntityDescription entityForName:@"SearchEngine" inManagedObjectContext:context]];
    engines = [context executeFetchRequest:request error:NULL];
    [request release];
    
    SRSearchEngine* engine;
    enumerator = [engines objectEnumerator];
    while (engine = [enumerator nextObject]) {
        // Get URL string
        NSString*   urlString;
        urlString = [SRSearchEngineIconDatabase urlStringForIconWithSearchEngine:engine];
        if (!urlString) {
            continue;
        }
        
        // Check icon existence
        if ([self iconForURLString:urlString]) {
            continue;
        }
        if ([[[WebIconDatabase sharedIconDatabase] iconURLForURL:urlString] length] > 0) {
            // Store it in database
            NSImage*    image;
            image = [[WebIconDatabase sharedIconDatabase] 
                    iconForURL:urlString withSize:NSMakeSize(16, 16)];
            [self setIcon:image forURLStirng:urlString];
            
            continue;
        }
        
        // Load icon
        WebView*    webView;
        webView = [[WebView alloc] initWithFrame:NSZeroRect];
        [webView setFrameLoadDelegate:self];
        [[webView mainFrame] loadRequest:
                [NSURLRequest requestWithURL:[NSURL URLWithString:urlString]]];
    }
}

- (void)saveDatabase
{
    // Get path
    NSString*   path;
    path = [self _searchEnginIconPath];
    if (!path) {
        return;
    }
    
    // Convert image to data
    NSMutableDictionary*    dict;
    NSEnumerator*           enumerator;
    NSString*               urlString;
    dict = [NSMutableDictionary dictionary];
    enumerator = [_iconDict keyEnumerator];
    while (urlString = [enumerator nextObject]) {
        NSImage*    image;
        NSData*     data;
        image = [_iconDict objectForKey:urlString];
        data = [image TIFFRepresentation];
        
        [dict setObject:data forKey:urlString];
    }
    
    // Save file
    [dict writeToFile:path atomically:YES];
}

//--------------------------------------------------------------//
#pragma mark -- WebView frame load delegate --
//--------------------------------------------------------------//

- (void)webView:(WebView*)webView didReceiveIcon:(NSImage*)image forFrame:(WebFrame*)frame
{
    // Get URL string
    NSString*   urlString;
    urlString = [[[[frame dataSource] initialRequest] URL] absoluteString];
    if (!urlString) {
        return;
    }
    
    // Set icon
    [self setIcon:image forURLStirng:urlString];
}

- (void)webView:(WebView*)webView didFinishLoadForFrame:(WebFrame*)frame
{
    // Release web view
    [webView setFrameLoadDelegate:nil];
//    [webView stopLoading:self];
//    [webView _close];
//    [webView release];
}

- (void)webView:(WebView*)webView didFailLoadWithError:(NSError*)error forFrame:(WebFrame*)frame
{
    // Release web view
    [webView setFrameLoadDelegate:nil];
//    [webView stopLoading:self];
//    [webView _close];
//    [webView release];
}

@end
