/*
SRStatusBar.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRStatusBar.h"

static NSImage* _statusBarImage = nil;
static NSImage* _statusBarEdgeImage = nil;
static NSRect   _statusBarRect = {{0, 0}, {0, 0}};
static NSRect   _statusBarEdgeRect = {{0, 0}, {0, 0}};

static int      _buttonMargin = 18;
static int      _statusIconButtonMargin = 4;

@implementation SRStatusBar

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (void)load
{
    NSAutoreleasePool*  pool;
    pool = [[NSAutoreleasePool alloc] init];
    
    // Get resources
    if (!_statusBarImage) {
        _statusBarImage = [[NSImage imageNamed:@"statusBar"] retain];
        _statusBarEdgeImage = [[NSImage imageNamed:@"statusBarEdge"] retain];
        
        _statusBarRect.size = [_statusBarImage size];
        _statusBarEdgeRect.size = [_statusBarEdgeImage size];
    }
    
    [pool release];
}

- (void)_init
{
    // Get frame
    NSRect  frame;
    frame = [self frame];
    
    // Create separator
    NSRect  separatorFrame;
    separatorFrame.origin.x = 0;
    separatorFrame.origin.y = frame.size.height - 1;
    separatorFrame.size.width = frame.size.width;
    separatorFrame.size.height = 1;
    
    _separator = [[NSBox alloc] initWithFrame:separatorFrame];
    [_separator setBoxType:NSBoxSeparator];
    [_separator setAutoresizingMask:NSViewWidthSizable | NSViewMaxYMargin];
    [self addSubview:_separator];
    [_separator release];
    
    // Create status text field
    NSRect  statusFrame;
    statusFrame.origin.x = 4;
    statusFrame.origin.y = 4;
    statusFrame.size.width = frame.size.width;
    statusFrame.size.height = 14;
    
    _statusTextField = [[HMEtchedTextField alloc] initWithFrame:statusFrame];
    [_statusTextField setAutoresizingMask:NSViewWidthSizable | NSViewMaxYMargin];
    [_statusTextField setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
    [self addSubview:_statusTextField];
    [_statusTextField release];
    
    // Create buttons and icons array
    _buttons = [[NSMutableArray array] retain];
    _statusIcons = [[NSMutableArray array] retain];
}

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    // Common init
    [self _init];
    
    return self;
}

- (id)initWithCoder:(id)coder
{
    self = [super initWithCoder:coder];
    if (!self) {
        return nil;
    }
    
    // Common init
    [self _init];
    
    return self;
}

- (void)dealloc
{
    [_buttons release];
    [_statusIcons release];
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Status text filed management --
//--------------------------------------------------------------//

- (HMEtchedTextField*)statusTextField
{
    return _statusTextField;
}

//--------------------------------------------------------------//
#pragma mark -- Button and status icon management --
//--------------------------------------------------------------//

+ (NSSize)buttonSize
{
    return NSMakeSize(32, 22);
}

- (void)_updateStatusTextField
{
    // Get frame
    NSRect frame;
    frame = [self frame];
    
    NSEnumerator*   enumerator;
    NSButton*       button;
    
    // Decide status icon frame
    NSRect  statusIconFrame = NSZeroRect;
    enumerator = [_statusIcons objectEnumerator];
    while (button = [enumerator nextObject]) {
        statusIconFrame = NSUnionRect(statusIconFrame, [button frame]);
    }
    
    // Decide button frame
    NSRect  buttonFrame = NSZeroRect;
    enumerator = [_buttons objectEnumerator];
    while (button = [enumerator nextObject]) {
        buttonFrame = NSUnionRect(buttonFrame, [button frame]);
    }
    
    // Decide status text field frame
    NSRect  statusFrame;
    statusFrame = [_statusTextField frame];
    statusFrame.origin.x = statusIconFrame.origin.x + statusIconFrame.size.width + 4;
    statusFrame.size.width = buttonFrame.origin.x - statusFrame.origin.x;
    [_statusTextField setFrame:statusFrame];
}

- (void)_updateButtonsFrame
{
    // Get frame
    NSRect frame;
    frame = [self frame];
    
    // Decide position
    int x;
    x = frame.size.width - _buttonMargin;
    
    // Set buttons frame
    NSEnumerator*   enumerator;
    NSButton*       button;
    NSRect          buttonFrame;
    enumerator = [_buttons objectEnumerator];
    while (button = [enumerator nextObject]) {
        NSSize  buttonSize;
        buttonSize = [button frame].size;
        
        buttonFrame.origin.x = x - buttonSize.width;
        buttonFrame.origin.y = (frame.size.height - 1 - buttonSize.height) / 2;
        buttonFrame.size = buttonSize;
        [button setFrame:buttonFrame];
        
        x -= buttonFrame.size.width - 1;
    }
    
    // Move status text field
    [self _updateStatusTextField];
    
    // Update itself
    [self setNeedsDisplay:YES];
}

- (void)_updateStatusIconButtonsFrame
{
    // Get frame
    NSRect frame;
    frame = [self frame];
    
    // Decide position
    int x;
    x = _statusIconButtonMargin;
    
    // Set status icon buttons frame
    NSEnumerator*   enumerator;
    NSButton*       button;
    NSRect          buttonFrame;
    enumerator = [_statusIcons objectEnumerator];
    while (button = [enumerator nextObject]) {
        NSSize  buttonSize;
        buttonSize = [button frame].size;
        
        buttonFrame.origin.x = x;
        buttonFrame.origin.y = (frame.size.height - 1 - buttonSize.height) / 2;
        buttonFrame.size = buttonSize;
        [button setFrame:buttonFrame];
        
        x += buttonFrame.size.width + 1;
    }
    
    // Move status text field
    [self _updateStatusTextField];
    
    // Update itself
    [self setNeedsDisplay:YES];
}

- (NSButton*)buttonAtIndex:(int)index
{
    if (index > [_buttons count]) {
        return nil;
    }
    
    return [_buttons objectAtIndex:index];
}

- (void)addButton:(NSButton*)button
{
    // Check arugment
    if (!button || [_buttons containsObject:button]) {
        return;
    }
    
    // Add button
    [_buttons addObject:button];
    [button setAutoresizingMask:NSViewMinXMargin | NSViewMaxYMargin];
    [self addSubview:button];
    
    // Update button and status icon frame
    [self _updateButtonsFrame];
}

- (void)removeButton:(NSButton*)button
{
    // Check arugment
    if (!button || ![_buttons containsObject:button]) {
        return;
    }
    
    // Remove button
    if ([button superview] == self) {
        [button removeFromSuperview];
    }
    [_buttons removeObject:button];
    
    // Update button and status icon frame
    [self _updateButtonsFrame];
}

- (void)addStatusIconButton:(NSButton*)button
{
    // Check arugment
    if (!button || [_statusIcons containsObject:button]) {
        return;
    }
    
    // Add status icon button
    [_statusIcons addObject:button];
    [button setAutoresizingMask:NSViewMaxXMargin | NSViewMaxYMargin];
    if ([button superview] && [button superview] != self) {
        [self removeFromSuperview];
    }
    [self addSubview:button];
    
    // Update status icon frame
    [self _updateStatusIconButtonsFrame];
}

- (void)removeStatusIconButton:(NSButton*)button
{
    // Check arugment
    if (!button || ![_statusIcons containsObject:button]) {
        return;
    }
    
    // Remove status icon button
    if ([button superview] == self) {
        [button removeFromSuperview];
    }
    [_statusIcons removeObject:button];
    
    // Update status icon frame
    [self _updateStatusIconButtonsFrame];
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (void)drawRect:(NSRect)rect
{
    // Get frame
    NSRect  frame;
    frame = [self frame];
    
    // Draw background
    if (NSIntersectsRect(frame, rect)) {
        [_statusBarImage drawInRect:frame fromRect:_statusBarRect operation:NSCompositeCopy fraction:1.0f];
    }
    
    // Draw status bar edge
    NSRect  edgeRect;
    edgeRect.origin.x = frame.size.width - 1;
    edgeRect.origin.y = 0;
    edgeRect.size = _statusBarEdgeRect.size;
    [_statusBarEdgeImage drawInRect:edgeRect fromRect:_statusBarEdgeRect 
            operation:NSCompositeSourceOver fraction:1.0f];
    
    NSRect          buttonRect = NSZeroRect;
    NSEnumerator*   enumerator;
    NSButton*       button;
    enumerator = [_buttons objectEnumerator];
    while (button = [enumerator nextObject]) {
        buttonRect = NSUnionRect(buttonRect, [button frame]);
    }
    
    if (!NSIsEmptyRect(buttonRect)) {
        edgeRect.origin.x = buttonRect.origin.x - 1;
        edgeRect.origin.y = 0;
        edgeRect.size = _statusBarEdgeRect.size;
        [_statusBarEdgeImage drawInRect:edgeRect fromRect:_statusBarEdgeRect 
                operation:NSCompositeSourceOver fraction:1.0f];
        
        edgeRect.origin.x = buttonRect.origin.x + buttonRect.size.width;
        edgeRect.origin.y = 0;
        edgeRect.size = _statusBarEdgeRect.size;
        [_statusBarEdgeImage drawInRect:edgeRect fromRect:_statusBarEdgeRect 
                operation:NSCompositeSourceOver fraction:1.0f];
    }
}

@end
