/*
HMSplitView.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMSplitView.h"

static NSColor* _lineColor = nil;

@implementation HMSplitView

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (void)_init
{
    // Initialize instance variables
    _type = -1;
}

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    // Common init
    [self _init];
    
    return self;
}

- (id)initWithCoder:(NSCoder*)coder
{
    self = [super initWithCoder:coder];
    if (!self) {
        return nil;
    }
    
    // Common init
    [self _init];
    
    return self;
}

//--------------------------------------------------------------//
#pragma mark -- Accessors --
//--------------------------------------------------------------//

- (int)splitType
{
    return _type;
}

- (void)setSplitType:(int)type
{
    if (_type == type) {
        return;
    }
    
    _type = type;
    
    // Adjust subviews
    [self adjustSubviews];
}

- (void)adjustSubviews
{
    [super adjustSubviews];
    
    BOOL    isResized = NO;
    
    // Get frame
    NSRect  frame;
    frame = [self frame];
    
    // Get divider thickness
    float   divider;
    divider = [self dividerThickness];
    
    // Get delegate
    id  delegate;
    delegate = [self delegate];
    
    // For vertical
    if ([self isVertical]) {
        // Set subview frames
        NSArray*    subviews;
        subviews = [self subviews];
        
        NSView* subview;
        NSRect  subframe;
        float   x = 0;
        
        int i;
        for (i = 0; i < [subviews count] - 1; i++) {
            // Get subview and subframe
            subview = [subviews objectAtIndex:i];
            subframe = [subview frame];
            
            // Get proposed value
            float   value, min, max;
            value = min = max = [subview frame].size.width;
            if ([delegate respondsToSelector:@selector(splitView:constrainMinCoordinate:ofSubviewAt:)]) {
                min = [delegate splitView:self constrainMinCoordinate:min ofSubviewAt:i];
            }
            if ([delegate respondsToSelector:@selector(splitView:constrainMaxCoordinate:ofSubviewAt:)]) {
                max = [delegate splitView:self constrainMaxCoordinate:max ofSubviewAt:i];
            }
            
            float   newValue = value;
            if (value < min) {
                newValue = min;
            }
            if (value > max) {
                newValue = max;
            }
            
            // Update subframe
            if (value != newValue || x != subframe.origin.x) {
                subframe.origin.x = x;
                subframe.size.width = newValue;
                
                [subview setFrame:subframe];
                [subview setNeedsDisplay:YES];
                
                isResized = YES;
            }
            
            // Increase x
            x += newValue + divider;
        }
        
        subview = [subviews lastObject];
        subframe = [subview frame];
        
        if (subframe.origin.x != x) {
            subframe.origin.x = x;
            subframe.size.width = frame.size.width - x;
            
            [subview setFrame:subframe];
            [subview setNeedsDisplay:YES];
            
            isResized = YES;
        }
    }
    
    // For horizontal
    else {
    }
    
    // Notify to delegate
    if (isResized) {
        if ([delegate respondsToSelector:@selector(splitViewDidResizeSubviews:)]) {
            [delegate splitViewDidResizeSubviews:
                    [NSNotification notificationWithName:NSSplitViewDidResizeSubviewsNotification 
                            object:self]];
        }
    }
}

//--------------------------------------------------------------//
#pragma mark -- Dragging --
//--------------------------------------------------------------//

- (void)mouseDown:(NSEvent*)event
{
    // Get delegate
    id  delegate;
    delegate = [self delegate];
    
    // Notify to delegate
    if (delegate && [delegate respondsToSelector:@selector(splitViewStartDragging:)]) {
        [delegate splitViewStartDragging:self];
    }
    
    // Invoke super
    [super mouseDown:event];
    
    // Notify to delegate
    if (delegate && [delegate respondsToSelector:@selector(splitViewEndDragging:)]) {
        [delegate splitViewEndDragging:self];
    }
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (float)dividerThickness
{
    switch (_type) {
    case HMSplitLine: return 1.0f;
    case HMSplitWide: return 10.0f;
    }
    
    return [super dividerThickness];
}

- (void)_drawLineDividerInRect:(NSRect)rect
{
    if (!_lineColor) {
        _lineColor = [[NSColor colorWithCalibratedWhite:0.624f alpha:1.0f] retain];
    }

    // Draw line
    [_lineColor set];
    NSRectFill(rect);
}

- (void)_drawWideDividerInRect:(NSRect)rect
{
    // Draw background
    NSImage*    backImage;
    backImage = [NSImage imageNamed:@"splitWideDividerBack"];
    [backImage setFlipped:[self isFlipped]];
    
    NSRect  srcRect, destRect;
    srcRect.origin = NSZeroPoint;
    srcRect.size = [backImage size];
    [backImage drawInRect:rect fromRect:srcRect operation:NSCompositeSourceOver fraction:1.0f];
    
    // Draw knob
    NSImage*    knobImage;
    knobImage = [NSImage imageNamed:@"splitWideDividerKnob"];
    [knobImage setFlipped:[self isFlipped]];
    
    srcRect.origin = NSZeroPoint;
    srcRect.size = [knobImage size];
    destRect.origin.x = rect.origin.x + floor(rect.size.width / 2 - srcRect.size.width / 2);
    destRect.origin.y = rect.origin.y;
    destRect.size = srcRect.size;
    [knobImage drawInRect:destRect fromRect:srcRect operation:NSCompositeSourceOver fraction:1.0f];
}

- (void)drawDividerInRect:(NSRect)rect
{
    switch (_type) {
    case HMSplitLine: {
        [self _drawLineDividerInRect:rect];
        break;
    }
    case HMSplitWide: {
        [self _drawWideDividerInRect:rect];
        break;
    }
    }
}

@end
