package test.org.seasar.transaction;

import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.xa.XAResource;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.seasar.transaction.DefaultXAResource;
import org.seasar.transaction.TransactionImpl;
import org.seasar.transaction.XAResourceManager;
import org.seasar.transaction.XidImpl;
import org.seasar.util.SeasarRuntimeException;

public class TransactionImplTest extends TestCase {

    private TransactionImpl _tx;

    public TransactionImplTest(String name) {
        super(name);
    }

    public void testEquals() throws Exception {
        assertEquals("1", _tx, _tx);
        assertTrue("2", !_tx.equals(null));
        assertTrue("3", !_tx.equals(new TransactionImpl(new XidImpl())));
        assertTrue("4", !_tx.equals("test"));

        XidImpl xid2 = new XidImpl();
        TransactionImpl tx2 = new TransactionImpl(xid2);
        TransactionImpl tx3 = new TransactionImpl(xid2);
        assertEquals("5", tx2, tx3);
        System.out.println(tx2);
        System.out.println(tx3);
    }

    public void testBegin() throws Exception {
        DefaultXAResource xaRes = new DefaultXAResource();
        XAResourceManager.getInstance().addXAResource(xaRes);
        assertEquals("1", 1, XAResourceManager.getInstance().getXAResourceCount());
        _tx.begin();
        assertEquals("2", DefaultXAResource.RS_ACTIVE, xaRes.getStatus());
        assertEquals("3", 0, XAResourceManager.getInstance().getXAResourceCount());
    }

    public void testSuspend() throws Exception {
        DefaultXAResource xaRes = new DefaultXAResource();
        XAResourceManager.getInstance().addXAResource(xaRes);
        _tx.begin();
        _tx.suspend();
        assertEquals("1", DefaultXAResource.RS_SUSPENDED, xaRes.getStatus());
        try {
            _tx.suspend();
            fail("2");
        } catch (IllegalStateException ex) {
            System.out.println(ex);
        }
        _tx.resume();
        _tx.commit();
        try {
            _tx.suspend();
            fail("3");
        } catch (IllegalStateException ex) {
            System.out.println(ex);
        }
    }


    public void testResume() throws Exception {
        DefaultXAResource xaRes = new DefaultXAResource();
        XAResourceManager.getInstance().addXAResource(xaRes);
        _tx.begin();
        _tx.suspend();
        _tx.resume();
        assertEquals("1", DefaultXAResource.RS_ACTIVE, xaRes.getStatus());
        try {
            _tx.resume();
            fail("2");
        } catch (IllegalStateException ex) {
            System.out.println(ex);
        }
    }

    public void testOnePhaseCommit() throws Exception {
        DefaultXAResource xaRes = new DefaultXAResource();
        XAResourceManager.getInstance().addXAResource(xaRes);
        Sync sync = new Sync();
        _tx.begin();
        _tx.registerSynchronization(sync);
        _tx.commit();
        assertTrue("1", sync._beforeCompleted);
        assertTrue("2", sync._afterCompleted);
        assertEquals("3", Status.STATUS_COMMITTED, sync._completedStatus);
        assertEquals("4", DefaultXAResource.RS_NONE, xaRes.getStatus());
    }


    public void testTwoPhaseCommit() throws Exception {
        DefaultXAResource xaRes = new DefaultXAResource();
        DefaultXAResource xaRes2 = new DefaultXAResource();
        XAResourceManager.getInstance().addXAResource(xaRes);
        XAResourceManager.getInstance().addXAResource(xaRes2);
        _tx.begin();
        _tx.commit();
        assertEquals("1", Status.STATUS_COMMITTED, _tx.getStatus());
        assertEquals("2", DefaultXAResource.RS_NONE, xaRes.getStatus());
        assertEquals("3", DefaultXAResource.RS_NONE, xaRes2.getStatus());
    }

    public void testCommitForException() throws Exception {
        DefaultXAResource xaRes = new DefaultXAResource();
        XAResourceManager.getInstance().addXAResource(xaRes);
        _tx.begin();
        _tx.suspend();
        try {
            _tx.commit();
            fail("1");
        } catch (IllegalStateException ex) {
            System.out.println(ex);
        }
        _tx.resume();
        _tx.commit();
        try {
            _tx.commit();
            fail("2");
        } catch (IllegalStateException ex) {
            System.out.println(ex);
        }
    }

    public void testRollback() throws Exception {
        DefaultXAResource xaRes = new DefaultXAResource();
        XAResourceManager.getInstance().addXAResource(xaRes);
        Sync sync = new Sync();
        _tx.begin();
        _tx.registerSynchronization(sync);
        _tx.rollback();
        assertTrue("1", sync._beforeCompleted);
        assertTrue("2", sync._afterCompleted);
        assertEquals("3", Status.STATUS_ROLLEDBACK, sync._completedStatus);
        assertEquals("4", DefaultXAResource.RS_NONE, xaRes.getStatus());
    }

    public void testRollbackForException() throws Exception {
        DefaultXAResource xaRes = new DefaultXAResource();
        XAResourceManager.getInstance().addXAResource(xaRes);
        _tx.begin();
        _tx.suspend();
        try {
            _tx.rollback();
            fail("1");
        } catch (IllegalStateException ex) {
            System.out.println(ex);
        }
        _tx.resume();
        _tx.rollback();
        try {
            _tx.rollback();
            fail("2");
        } catch (IllegalStateException ex) {
            System.out.println(ex);
        }
    }

    public void testSetRollbackOnly() throws Exception {
        _tx.begin();
        _tx.setRollbackOnly();
        assertEquals("1", Status.STATUS_MARKED_ROLLBACK, _tx.getStatus());
    }

    public void testSetRollbackOnlyForException() throws Exception {
        _tx.begin();
        _tx.commit();
        try {
            _tx.setRollbackOnly();
            fail("1");
        } catch (IllegalStateException ex) {
            System.out.println(ex);
        }
    }

    public void testDelistResource() throws Exception {
        DefaultXAResource xaRes = new DefaultXAResource();
        XAResourceManager.getInstance().addXAResource(xaRes);
        _tx.begin();
        _tx.delistResource(xaRes, XAResource.TMSUCCESS);
        assertEquals("1", DefaultXAResource.RS_SUCCESS, xaRes.getStatus());
    }

    public void testDelistResourceForException() throws Exception {
        _tx.begin();
        DefaultXAResource xaRes = new DefaultXAResource();
        try {
            _tx.delistResource(xaRes, XAResource.TMSUCCESS);
        } catch (SeasarRuntimeException ex) {
            System.out.println(ex);
        }
        _tx.suspend();
        _tx.resume();
        _tx.commit();
        try {
            _tx.delistResource(xaRes, XAResource.TMSUCCESS);
        } catch (IllegalStateException ex) {
            System.out.println(ex);
        }
    }


    protected void setUp() throws Exception {
        _tx = new TransactionImpl(new XidImpl());
    }

    protected void tearDown() throws Exception {
    }

    public static Test suite() {
        return new TestSuite(TransactionImplTest.class);
    }

    public static void main(String[] args) {
        junit.textui.TestRunner.main(new String[]{TransactionImplTest.class.getName()});
    }

    private class Sync implements Synchronization {

        private boolean _beforeCompleted = false;
        private boolean _afterCompleted = false;
        private int _completedStatus = Status.STATUS_UNKNOWN;

        public void beforeCompletion() {
            _beforeCompleted = true;
        }

        public void afterCompletion(int status) {
            _afterCompleted = true;
            _completedStatus = status;
        }
    }
}
