package org.seasar.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;
import java.util.jar.JarFile;

public final class ResourceUtil {

    private ResourceUtil() { }

    public static boolean isExist(final String name) {
        return Thread.currentThread().getContextClassLoader().getResourceAsStream(resolveName(name)) != null;
    }

    public static String resolveName(final String name) {
        return resolveName(name, null);
    }

    public static String resolveName(String name, final String extension) {
        String ext = extension != null ? "." + extension : "";
        if (name.indexOf('/') >= 0) {
            if (name.charAt(0) == '/') {
                name = name.substring(1);
            }
            if (name.indexOf('.') > 0) {
                return name;
            } else {
                return name + ext;
            }
        } else {
            if (extension != null) {
            	if (name.endsWith(extension)) {
                	name = trimExtension(name, extension);
            	}
            } else {
                int pos = name.indexOf('.');
                int pos2 = name.lastIndexOf('.');
                if (pos > 0 && pos == pos2) {
                    return name;
                }
            }
            return name.replace('.', '/') + ext;
        }
    }

    public static String unresolveName(String name, final String extension) {
        if (name.charAt(0) == '/') {
            name = name.substring(1);
        }
        return trimExtension(name, extension).replace('/', '.');
    }

    public static String trimExtension(final String name, final String extension) {
        int extPos = name.lastIndexOf("." + extension);
        if (extPos >= 0) {
            return name.substring(0, extPos);
        }
        int lastPeriodPos = name.lastIndexOf('.');
        if (lastPeriodPos < 0) {
            return name;
        }
        int lastPeriodPos2 = name.lastIndexOf('.', lastPeriodPos - 1);
        if (lastPeriodPos2 < 0) {
            return name.substring(0, lastPeriodPos);
        }
        return name;
    }

    public static InputStream getResourceAsStream(final String name) {
        return getResourceAsStream(name, null);
    }

    public static InputStream getResourceAsStream(final String name, final String extension) {
        InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(resolveName(name, extension));
        if (is != null) {
            return is;
        } else {
            throw new SeasarRuntimeException("ESSR0001", new Object[]{resolveName(name, extension)});
        }
    }
    
    public static URL getResource(final String name) {
        return getResource(name, null);
    }
    
    public static URL getResource(final String name, final String extension) {
        URL url = Thread.currentThread().getContextClassLoader().getResource(resolveName(name, extension));
        if (url != null) {
            return url;
        } else {
            throw new SeasarRuntimeException("ESSR0001", new Object[]{resolveName(name, extension)});
        }
    }
    
	public static final File getFile(String path) {
		return getFile(path, null);
	}
	
	public static final File getFile(String path, String extension) {
		URL url = ResourceUtil.getResource(path, extension);
		Assertion.assertFound("path", url);
		return new File(url.getFile());
	}

    
    public static long getLastModified(URL url) {
    	String fileName = url.getFile();
    	if (fileName.startsWith("file:")) {
    		int pos = fileName.indexOf('!');
    		fileName = fileName.substring(5, pos);
    	}
    	return new File(fileName).lastModified();
    }

    public static Properties getProperties(final String name) {
        if (name.charAt(0) == '/') {
            return getProperties(name, null);
        } else {
            return getProperties(name, "properties");
        }
    }

    public static Properties getProperties(final String name, final String extension) {
        Properties props = new Properties();
        InputStream is = getResourceAsStream(name, extension);
        try {
            props.load(is);
            return props;
        } catch (IOException ex) {
            throw new SeasarRuntimeException("ESSR0008", new Object[]{resolveName(name, extension), ex}, ex);
        }
    }

    public static File getFileFromClassPath(final String name) {
        String classPath = System.getProperty("java.class.path");
        int pos = classPath.indexOf(name);
        if (pos < 0) {
            throw new SeasarRuntimeException("ESSR0009", new Object[]{name, classPath});
        }
        int start = classPath.lastIndexOf(System.getProperty("path.separator"), pos) + 1;
        int end = classPath.indexOf(System.getProperty("path.separator"), pos);
        String location = end > 0 ? classPath.substring(start, end) : classPath.substring(start);
        return new File(location);
    }

    public static boolean isJarFile(final File file) {
        try {
            new JarFile(file, true);
            return true;
        } catch (Exception ex) {
            return false;
        }
    }

    public static boolean isAvailableDirectory(final File file) {
        return file.isDirectory() && file.exists() && file.canRead();
    }
    
    public static void close(InputStream is) {
    	try {
        	is.close();
    	} catch (IOException ex) {
    		throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
    	}
    }
    
    public static final byte[] getBytes(File file) {
		FileInputStream fis = null;
		byte[] bytes = null;
		byte[] buf = new byte[8192];
		try {
			fis = new FileInputStream(file);
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			int n = 0;
			while((n = fis.read(buf, 0, buf.length)) != -1){
				baos.write(buf, 0, n);
	        }
			bytes = baos.toByteArray();
		} catch (IOException ex) {
			throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
		} finally {
			if (fis != null) {
				close(fis);
			}
		}
		return bytes;
	}
	
	public static String getExtension(final String name) {
        int extPos = name.lastIndexOf(".");
        if (extPos >= 0) {
        	return name.substring(extPos + 1);
        } else {
        	return "";
        }
    }
}