package org.seasar.system;

import org.seasar.log.Logger;
import org.seasar.timer.TimeoutManager;
import org.seasar.timer.TimeoutTarget;
import org.seasar.timer.TimeoutTask;
import org.seasar.util.ELinkedList;
import org.seasar.util.SeasarException;
import org.seasar.util.SeasarRuntimeException;

public class ThreadPool {

    private static Logger _logger = Logger.getLogger(ThreadPool.class);
    private ELinkedList _idlePool = new ELinkedList();
    private int _activeSize = 0;
    private int _maxSize = 50;
    private int _minSize = 5;
    private int _shrinkSize = 10;
    private int _shrinkInterval = 10 * 60;
    private Shrinker _shrinker;

    public ThreadPool() {
    }

    public ThreadPool(final int maxSize) {
        setMaxSize(maxSize);
    }

    public void start() throws SeasarException {
        _activeSize = 0;
        for (int i = 0; i < _minSize; i++) {
            _idlePool.addLast(new Worker());
        }
        _shrinker = new Shrinker();
    }

    public synchronized void stop() {
        if (_shrinker != null) {
            _shrinker.cancel();
            _shrinker = null;
        }
        final long tenSec = 10 * 1000;
        while (_activeSize > 0) {
            int activeSizeBeforeWait = _activeSize;
            try {
                wait(tenSec);
            } catch (InterruptedException ignore) {
            }
            if (activeSizeBeforeWait == _activeSize) {
                _activeSize--;
            }
        }
        while (_idlePool.size() > 0) {
            Worker worker = (Worker) _idlePool.removeFirst();
            worker.die();
        }
    }


    public int getMaxSize() {
        return _maxSize;
    }


    public void setMaxSize(final int maxSize) {
        if (maxSize < 1) {
            throw new SeasarRuntimeException("ESSR0027", new Object[]{String.valueOf(maxSize)});
        }
        _maxSize = maxSize;
        if (_minSize > maxSize) {
            _minSize = maxSize;
        }
        if (_shrinkSize > maxSize) {
            _shrinkSize = maxSize;
        }
    }

    public int getMinSize() {
        return _minSize;
    }

    public void setMinSize(final int minSize) {
        if (minSize < 0) {
            throw new SeasarRuntimeException("ESSR0027", new Object[]{String.valueOf(minSize)});
        }
        _minSize = minSize;
        if (_maxSize < minSize) {
            _maxSize = minSize;
        }
        if (_shrinkSize < minSize) {
            _shrinkSize = minSize;
        }
    }

    public int getShrinkSize() {
        return _shrinkSize;
    }

    public void setShrinkSize(final int shrinkSize) {
        if (shrinkSize < 0 || shrinkSize < _minSize || _maxSize < shrinkSize) {
            throw new SeasarRuntimeException("ESSR0027", new Object[]{String.valueOf(shrinkSize)});
        }
        _shrinkSize = shrinkSize;
    }


    public int getShrinkInterval() {
        return _shrinkInterval;
    }


    public void setShrinkInterval(final int shrinkInterval) {
        _shrinkInterval = shrinkInterval;
    }

    public int getActiveSize() {
        return _activeSize;
    }

    public int getIdleSize() {
        return _idlePool.size();
    }

    public void request(final Runnable work) {
        Worker worker = requestWorker();
        worker.request(work);
    }

    public synchronized Worker requestWorker() {
        while (_activeSize >= _maxSize) {
            try {
                wait();
            } catch (InterruptedException ignore) {
            }
        }
        Worker worker = null;
        if (_idlePool.size() == 0) {
            worker = new Worker();
        } else {
            worker = (Worker) _idlePool.removeLast();
        }
        _activeSize++;
        return worker;
    }

    private synchronized void returnWorker(final Worker worker) {
        _activeSize--;
        _idlePool.addLast(worker);
        notify();
    }

    public final class Worker implements Runnable {

        private Thread _thread;
        private boolean _running = true;
        private boolean _synchronizing = false;
        private Runnable _work;


        Worker() {
            _thread = new Thread(this);
            _thread.setDaemon(true);
            _thread.start();
        }

        public synchronized void die() {
            _running = false;
            notify();
        }

        public synchronized void request(final Runnable work) {
            if (_work != null) {
                throw new IllegalStateException("Worker already has work");
            }
            _work = work;
            notify();
        }

        public void synchronize() {
            if (_work == null) {
                return;
            }
            try {
                synchronized (_work) {
                    _synchronizing = true;
                    try {
                    	_work.wait();
                    } catch (InterruptedException ignore) {
                    }
                }
            } catch (NullPointerException ignore) {
            }
        }

        public void run() {
            while (_running) {
                synchronized (this) {
                    if (_work != null) {
                        try {
                            _work.run();
                        } catch (Throwable t) {
                            _logger.error(t);
                        }
                        try {
                            if (_synchronizing) {
                                synchronized (_work) {
                                    _synchronizing = false;
                                    _work.notify();
                                }
                            }
                        } finally {
                            _work = null;
                            returnWorker(this);
                        }
                    }
                    while (_running && _work == null) {
                        try {
                            wait();
                        } catch (InterruptedException ignore) {
                        }
                    }
                }
            }
        }
    }


    private class Shrinker implements TimeoutTarget {

        private TimeoutTask _timeoutTask;


        Shrinker() {
            _timeoutTask = TimeoutManager.getInstance().addTimeoutTarget(
                    this, _shrinkInterval, true);
        }

        public void expired() {
            synchronized (_idlePool) {
                while (_idlePool.size() > _shrinkSize) {
                    Worker worker = (Worker) _idlePool.removeFirst();
                    worker.die();
                }
            }
        }


        void cancel() {
            _timeoutTask.cancel();
            _timeoutTask = null;
        }
    }
}
