package org.maachang.rimdb.util;

import java.util.Collection;

import org.maachang.rimdb.RimDbException;

/**
 * 変換処理系.
 * 
 * @version 2014/07/07
 * @author masahito suzuki
 * @since rimdb-1.00
 */
@SuppressWarnings("unchecked")
public final class ConvertUtil {
	protected ConvertUtil() {
	};

	/** char文字のチェックを行う配列. **/
	public static final byte[] CHECK_CHAR = new byte[65536];
	static {
		// スペース系は１.
		// ドットは２.
		// 数字の終端文字は３.
		CHECK_CHAR[' '] = 1;
		CHECK_CHAR['\t'] = 1;
		CHECK_CHAR['\r'] = 1;
		CHECK_CHAR['\n'] = 1;
		CHECK_CHAR['.'] = 2;
		CHECK_CHAR['L'] = 3;
		CHECK_CHAR['l'] = 3;
		CHECK_CHAR['F'] = 3;
		CHECK_CHAR['f'] = 3;
		CHECK_CHAR['D'] = 3;
		CHECK_CHAR['d'] = 3;
	}

	/**
	 * 文字列内容が数値かチェック.
	 * 
	 * @param num
	 *            対象のオブジェクトを設定します.
	 * @return boolean [true]の場合、文字列内は数値が格納されています.
	 */
	public static final boolean isNumeric(Object num) {
		if (num == null) {
			return false;
		} else if (num instanceof Number) {
			return true;
		} else if (!(num instanceof String)) {
			num = num.toString();
		}
		char c;
		String s = (String) num;
		int i, start, end, flg, dot;
		start = flg = 0;
		dot = -1;
		end = s.length() - 1;

		for (i = start; i <= end; i++) {
			c = s.charAt(i);
			if (flg == 0 && CHECK_CHAR[c] != 1) {
				if (c == '-') {
					start = i + 1;
				} else {
					start = i;
				}
				flg = 1;
			} else if (flg == 1 && CHECK_CHAR[c] != 0) {
				if (c == '.') {
					if (dot != -1) {
						return false;
					}
					dot = i;
				} else {
					end = i - 1;
					break;
				}
			}
		}
		if (flg == 0) {
			return false;
		}
		if (start <= end) {
			for (i = start; i <= end; i++) {
				if (!((c = s.charAt(i)) == '.' || (c >= '0' && c <= '9'))) {
					return false;
				}
			}
		} else {
			return false;
		}
		return true;
	}

	/**
	 * 対象文字列が存在するかチェック.
	 * 
	 * @param v
	 *            対象の情報を設定します.
	 * @return boolean [true]の場合、文字列が存在します.
	 */
	public static final boolean useString(Object v) {
		if (v == null) {
			return false;
		}
		if (v instanceof CharSequence) {
			CharSequence cs = (CharSequence) v;
			if (cs.length() > 0) {
				int len = cs.length();
				for (int i = 0; i < len; i++) {
					if (CHECK_CHAR[cs.charAt(i)] == 1) {
						continue;
					}
					return true;
				}
			}
			return false;
		} else if (v instanceof Collection) {
			return !((Collection) v).isEmpty();
		}
		return true;
	}

	/**
	 * Boolean変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 */
	public static final Boolean convertBool(Object o) {
		if (o == null) {
			return null;
		}
		if (o instanceof Boolean) {
			return (Boolean) o;
		}
		if (o instanceof Number) {
			return (((Number) o).intValue() == 0) ? false : true;
		}
		if (o instanceof String) {
			return parseBoolean((String) o);
		}
		throw new RimDbException("BOOL型変換に失敗しました[" + o + "]");
	}

	/**
	 * Integer変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 */
	public static final Integer convertInt(Object o) {
		if (o == null) {
			return null;
		}
		if (o instanceof Integer) {
			return (Integer) o;
		} else if (o instanceof Number) {
			return ((Number) o).intValue();
		} else if (o instanceof String) {
			return parseInt((String) o);
		}
		throw new RimDbException("Int型変換に失敗しました[" + o + "]");
	}

	/**
	 * Long変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 */
	public static final Long convertLong(Object o) {
		if (o == null) {
			return null;
		}
		if (o instanceof Long) {
			return (Long) o;
		} else if (o instanceof Number) {
			return ((Number) o).longValue();
		} else if (o instanceof String) {
			return parseLong((String) o);
		}
		throw new RimDbException("Long型変換に失敗しました[" + o + "]");
	}

	/**
	 * Float変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 */
	public static final Float convertFloat(final Object o) {
		if (o == null) {
			return null;
		} else if (o instanceof Float) {
			return (Float) o;
		} else if (o instanceof Number) {
			return ((Number) o).floatValue();
		} else if (o instanceof String && isNumeric(o)) {
			return parseFloat((String) o);
		}
		throw new RimDbException("Float型変換に失敗しました[" + o + "]");
	}

	/**
	 * Double変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 */
	public static final Double convertDouble(Object o) {
		if (o == null) {
			return null;
		}
		if (o instanceof Double) {
			return (Double) o;
		} else if (o instanceof Number) {
			return ((Number) o).doubleValue();
		} else if (o instanceof String) {
			return parseDouble((String) o);
		}
		throw new RimDbException("Double型変換に失敗しました[" + o + "]");
	}

	/**
	 * 文字列変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 */
	public static final String convertString(Object o) {
		if (o == null) {
			return null;
		}
		if (o instanceof String) {
			return (String) o;
		}
		return o.toString();
	}

	/** 日付のみ表現. **/
	private static final java.sql.Date _cDate(long d) {
		return _cDate(new java.util.Date(d));
	}

	@SuppressWarnings("deprecation")
	private static final java.sql.Date _cDate(java.util.Date n) {
		return new java.sql.Date(n.getYear(), n.getMonth(), n.getDate());
	}

	/**
	 * 日付変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 */
	public static final java.sql.Date convertSqlDate(Object o) {
		if (o == null) {
			return null;
		}
		if (o instanceof java.util.Date) {
			return _cDate(((java.util.Date) o));
		} else if (o instanceof Long) {
			return _cDate((Long) o);
		} else if (o instanceof Number) {
			return _cDate(((Number) o).longValue());
		} else if (o instanceof String) {
			if (isNumeric(o)) {
				return _cDate(parseLong((String) o));
			}
			return DateTimeUtil.getDate((String) o);
		}
		throw new RimDbException("java.sql.Date型変換に失敗しました[" + o + "]");
	}

	/** 時間のみ表現. **/
	private static final java.sql.Time _cTime(long d) {
		return _cTime(new java.util.Date(d));
	}

	@SuppressWarnings("deprecation")
	private static final java.sql.Time _cTime(java.util.Date n) {
		return new java.sql.Time(n.getHours(), n.getMinutes(), n.getSeconds());
	}

	/**
	 * 時間変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 */
	public static final java.sql.Time convertSqlTime(Object o) {
		if (o == null) {
			return null;
		}
		if (o instanceof java.util.Date) {
			return _cTime((java.util.Date) o);
		} else if (o instanceof Long) {
			return _cTime((Long) o);
		} else if (o instanceof Number) {
			return _cTime(((Number) o).longValue());
		} else if (o instanceof String) {
			if (isNumeric(o)) {
				return _cTime(parseLong((String) o));
			}
			return DateTimeUtil.getTime((String) o);
		}
		throw new RimDbException("java.sql.Time型変換に失敗しました[" + o + "]");
	}

	/**
	 * 日付時間変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 */
	public static final java.sql.Timestamp convertSqlTimestamp(Object o) {
		if (o == null) {
			return null;
		}
		if (o instanceof java.util.Date) {
			if (o instanceof java.sql.Timestamp) {
				return (java.sql.Timestamp) o;
			}
			return new java.sql.Timestamp(((java.util.Date) o).getTime());
		} else if (o instanceof Long) {
			return new java.sql.Timestamp((Long) o);
		} else if (o instanceof Number) {
			return new java.sql.Timestamp(((Number) o).longValue());
		} else if (o instanceof String) {
			if (isNumeric(o)) {
				return new java.sql.Timestamp(parseLong((String) o));
			}
			return DateTimeUtil.getTimestamp((String) o);
		}
		throw new RimDbException("java.sql.Timestamp型変換に失敗しました[" + o + "]");
	}

	/**
	 * 通常日付変換.
	 * 
	 * @param n
	 *            変換対象の条件を設定します.
	 * @return 変換された内容が返却されます.
	 * @exception Exception
	 *                例外.
	 */
	public static final java.util.Date convertDate(Object o) {
		if (o == null) {
			return null;
		}
		if (o instanceof java.util.Date) {
			return (java.util.Date) o;
		} else if (o instanceof Long) {
			return new java.util.Date((Long) o);
		} else if (o instanceof Number) {
			return new java.util.Date(((Number) o).longValue());
		} else if (o instanceof String) {
			if (isNumeric(o)) {
				return new java.sql.Timestamp(parseLong((String) o));
			}
			return DateTimeUtil.getTimestamp((String) o);
		}
		throw new RimDbException("java.util.Date型変換に失敗しました[" + o + "]");
	}

	/**
	 * 文字列から、Boolean型に変換.
	 * 
	 * @param s
	 *            対象の文字列を設定します.
	 * @return boolean Boolean型が返されます.
	 */
	public static final boolean parseBoolean(String s) {
		char c;
		int i, start, flg, len;

		start = flg = 0;
		len = s.length();

		for (i = start; i < len; i++) {
			c = s.charAt(i);
			if (flg == 0 && CHECK_CHAR[c] != 1) {
				start = i;
				flg = 1;
			} else if (flg == 1 && CHECK_CHAR[c] == 1) {
				len = i;
				break;
			}
		}
		if (flg == 0) {
			throw new RimDbException("Boolean変換に失敗しました:" + s);
		}

		if (isNumeric(s)) {
			return "0".equals(s) ? false : true;
		} else if (eqEng(s, start, len, "true") || eqEng(s, start, len, "t")
				|| eqEng(s, start, len, "on")) {
			return true;
		} else if (eqEng(s, start, len, "false") || eqEng(s, start, len, "f")
				|| eqEng(s, start, len, "off")) {
			return false;
		}
		throw new RimDbException("Boolean変換に失敗しました:" + s);
	}

	/**
	 * 文字列から、int型数値に変換.
	 * 
	 * @param num
	 *            対象の文字列を設定します.
	 * @return int int型で変換された数値が返されます.
	 */
	public static final int parseInt(final String num) {
		char c;
		boolean mFlg = false;
		int v, i, len, ret, end;

		ret = end = v = 0;
		len = num.length();

		for (i = end; i < len; i++) {
			c = num.charAt(i);
			if (v == 0 && CHECK_CHAR[c] != 1) {
				if (c == '-') {
					end = i + 1;
					mFlg = true;
				} else {
					end = i;
				}
				v = 1;
			} else if (v == 1 && CHECK_CHAR[c] != 0) {
				len = i;
				break;
			}
		}
		if (v == 0) {
			throw new RimDbException("Int数値変換に失敗しました:" + num);
		}

		v = 1;
		for (i = len - 1; i >= end; i--) {
			c = num.charAt(i);
			if (c >= '0' && c <= '9') {
				ret += (v * (c - '0'));
				v *= 10;
			} else {
				throw new RimDbException("Int数値変換に失敗しました:" + num);
			}
		}
		return mFlg ? ret * -1 : ret;
	}

	/**
	 * 文字列から、long型数値に変換.
	 * 
	 * @param num
	 *            対象の文字列を設定します.
	 * @return long long型で変換された数値が返されます.
	 */
	public static final long parseLong(final String num) {
		char c;
		boolean mFlg = false;
		long ret = 0L;
		int len, end, i, flg;

		end = flg = 0;
		len = num.length();

		for (i = end; i < len; i++) {
			c = num.charAt(i);
			if (flg == 0 && CHECK_CHAR[c] != 1) {
				if (c == '-') {
					end = i + 1;
					mFlg = true;
				} else {
					end = i;
				}
				flg = 1;
			} else if (flg == 1 && CHECK_CHAR[c] != 0) {
				len = i;
				break;
			}
		}
		if (flg == 0) {
			throw new RimDbException("Long数値変換に失敗しました:" + num);
		}

		long v = 1L;
		for (i = len - 1; i >= end; i--) {
			c = num.charAt(i);
			if (c >= '0' && c <= '9') {
				ret += (v * (long) (c - '0'));
				v *= 10L;
			} else {
				throw new RimDbException("Long数値変換に失敗しました:" + num);
			}
		}
		return mFlg ? ret * -1L : ret;
	}

	/**
	 * 文字列から、float型数値に変換.
	 * 
	 * @param num
	 *            対象の文字列を設定します.
	 * @return float float型で変換された数値が返されます.
	 */
	public static final float parseFloat(final String num) {
		char c;
		boolean mFlg = false;
		float ret = 0f;
		int end, len, flg, dot, i;

		end = flg = 0;
		dot = -1;
		len = num.length();

		for (i = end; i < len; i++) {
			c = num.charAt(i);
			if (flg == 0 && CHECK_CHAR[c] != 1) {
				if (c == '-') {
					end = i + 1;
					mFlg = true;
				} else {
					end = i;
				}
				flg = 1;
			} else if (flg == 1 && CHECK_CHAR[c] != 0) {
				if (c == '.') {
					if (dot != -1) {
						throw new RimDbException("Float数値変換に失敗しました:" + num);
					}
					dot = i;
				} else {
					len = i;
					break;
				}
			}
		}
		if (flg == 0) {
			throw new RimDbException("Float数値変換に失敗しました:" + num);
		}

		float v = 1f;
		if (dot == -1) {
			for (i = len - 1; i >= end; i--) {
				c = num.charAt(i);
				if (c >= '0' && c <= '9') {
					ret += (v * (float) (c - '0'));
					v *= 10f;
				} else {
					throw new RimDbException("Float数値変換に失敗しました:" + num);
				}
			}
			return mFlg ? ret * -1f : ret;
		} else {
			for (i = dot - 1; i >= end; i--) {
				c = num.charAt(i);
				if (c >= '0' && c <= '9') {
					ret += (v * (float) (c - '0'));
					v *= 10f;
				} else {
					throw new RimDbException("Float数値変換に失敗しました:" + num);
				}
			}
			float dret = 0f;
			v = 1f;
			for (i = len - 1; i > dot; i--) {
				c = num.charAt(i);
				if (c >= '0' && c <= '9') {
					dret += (v * (float) (c - '0'));
					v *= 10f;
				} else {
					throw new RimDbException("Float数値変換に失敗しました:" + num);
				}
			}
			return mFlg ? (ret + (dret / v)) * -1f : ret + (dret / v);
		}
	}

	/**
	 * 文字列から、double型数値に変換.
	 * 
	 * @param num
	 *            対象の文字列を設定します.
	 * @return double double型で変換された数値が返されます.
	 */
	public static final double parseDouble(final String num) {
		char c;
		boolean mFlg = false;
		double ret = 0d;
		int end, len, flg, dot, i;

		end = flg = 0;
		dot = -1;
		len = num.length();

		for (i = end; i < len; i++) {
			c = num.charAt(i);
			if (flg == 0 && CHECK_CHAR[c] != 1) {
				if (c == '-') {
					end = i + 1;
					mFlg = true;
				} else {
					end = i;
				}
				flg = 1;
			} else if (flg == 1 && CHECK_CHAR[c] != 0) {
				if (c == '.') {
					if (dot != -1) {
						throw new RimDbException("Double数値変換に失敗しました:" + num);
					}
					dot = i;
				} else {
					len = i;
					break;
				}
			}
		}
		if (flg == 0) {
			throw new RimDbException("Double数値変換に失敗しました:" + num);
		}

		double v = 1d;
		if (dot == -1) {
			for (i = len - 1; i >= end; i--) {
				c = num.charAt(i);
				if (c >= '0' && c <= '9') {
					ret += (v * (double) (c - '0'));
					v *= 10d;
				} else {
					throw new RimDbException("Double数値変換に失敗しました:" + num);
				}
			}
			return mFlg ? ret * -1d : ret;
		} else {
			for (i = dot - 1; i >= end; i--) {
				c = num.charAt(i);
				if (c >= '0' && c <= '9') {
					ret += (v * (double) (c - '0'));
					v *= 10d;
				} else {
					throw new RimDbException("Double数値変換に失敗しました:" + num);
				}
			}
			double dret = 0d;
			v = 1d;
			for (i = len - 1; i > dot; i--) {
				c = num.charAt(i);
				if (c >= '0' && c <= '9') {
					dret += (v * (double) (c - '0'));
					v *= 10d;
				} else {
					throw new RimDbException("Double数値変換に失敗しました:" + num);
				}
			}
			return mFlg ? (ret + (dret / v)) * -1d : ret + (dret / v);
		}
	}

	/**
	 * boolean群を定義.
	 * 
	 * @param v
	 *            boolean群を設定します.
	 * @return boolean[] boolean群が返却されます.
	 */
	public static final boolean[] getArray(boolean... v) {
		return v;
	}

	/**
	 * int群を定義.
	 * 
	 * @param v
	 *            int群を設定します.
	 * @return int[] int群が返却されます.
	 */
	public static final int[] getArray(int... v) {
		return v;
	}

	/**
	 * long群を定義.
	 * 
	 * @param v
	 *            long群を設定します.
	 * @return long[] long群が返却されます.
	 */
	public static final long[] getArray(long... v) {
		return v;
	}

	/**
	 * float群を定義.
	 * 
	 * @param v
	 *            float群を設定します.
	 * @return float[] float群が返却されます.
	 */
	public static final float[] getArray(float... v) {
		return v;
	}

	/**
	 * double群を定義.
	 * 
	 * @param v
	 *            double群を設定します.
	 * @return double[] double群が返却されます.
	 */
	public static final double[] getArray(double... v) {
		return v;
	}

	/**
	 * java.sql.Date群を定義.
	 * 
	 * @param v
	 *            java.sql.Date群を設定します.
	 * @return java.sql.java.sql.Date[] double群が返却されます.
	 */
	public static final java.sql.Date[] getArray(java.sql.Date... v) {
		return v;
	}

	/**
	 * java.sql.Time群を定義.
	 * 
	 * @param v
	 *            java.sql.Time群を設定します.
	 * @return java.sql.java.sql.Time[] double群が返却されます.
	 */
	public static final java.sql.Time[] getArray(java.sql.Time... v) {
		return v;
	}

	/**
	 * java.sql.Timestamp群を定義.
	 * 
	 * @param v
	 *            java.sql.Timestamp群を設定します.
	 * @return java.sql.java.sql.Timestamp[] double群が返却されます.
	 */
	public static final java.sql.Timestamp[] getArray(java.sql.Timestamp... v) {
		return v;
	}

	/**
	 * String群を定義.
	 * 
	 * @param v
	 *            String群を設定します.
	 * @return String[] String群が返却されます.
	 */
	public static final String[] getArray(String... v) {
		return v;
	}

	/**
	 * Comparable群を定義.
	 * 
	 * @param v
	 *            Comparable群を設定します.
	 * @return Comparable[] Comparable群が返却されます.
	 */
	public static final Comparable[] getArray(Comparable... v) {
		return v;
	}

	/**
	 * Object群を定義.
	 * 
	 * @param v
	 *            Object群を設定します.
	 * @return Object[] Object群が返却されます.
	 */
	public static final Object[] getArray(Object... v) {
		return v;
	}

	/**
	 * ビットマスク長を取得.
	 * 
	 * @param x
	 *            対象の値を設定します.
	 * @return int ビットマスク長が返却されます.
	 */
	public static final int bitMask(int x) {
		if (x < 1) {
			return 1;
		}
		x |= (x >> 1);
		x |= (x >> 2);
		x |= (x >> 4);
		x |= (x >> 8);
		x |= (x >> 16);
		x = (x & 0x55555555) + (x >> 1 & 0x55555555);
		x = (x & 0x33333333) + (x >> 2 & 0x33333333);
		x = (x & 0x0f0f0f0f) + (x >> 4 & 0x0f0f0f0f);
		x = (x & 0x00ff00ff) + (x >> 8 & 0x00ff00ff);
		x = (x & 0x0000ffff) + (x >> 16 & 0x0000ffff);
		return 1 << (((x & 0x0000ffff) + (x >> 16 & 0x0000ffff)) - 1);
	}

	/**
	 * 英字の大文字小文字を区別しない、バイトチェック.
	 * 
	 * @param s
	 *            比較の文字を設定します.
	 * @param d
	 *            比較の文字を設定します.
	 * @return boolean [true]の場合、一致します.
	 */
	public static final boolean oneEng(char s, char d) {
		return AZ_az.oneEq(s, d);
	}

	/**
	 * 英字の大文字小文字を区別せずにチェック.
	 * 
	 * @param src
	 *            比較元文字を設定します.
	 * @param dest
	 *            比較先文字を設定します.
	 * @return boolean [true]の場合、一致します.
	 */
	public static final boolean eqEng(final String src, final String dest) {
		return AZ_az.eq(src, dest);
	}

	/**
	 * 英字の大文字小文字を区別せずにチェック.
	 * 
	 * @param src
	 *            比較元文字を設定します.
	 * @param start
	 *            srcのチェック開始位置を設定します.
	 * @param end
	 *            srcのチェック終了位置を設定します.
	 * @param dest
	 *            比較先文字を設定します.
	 * @return boolean [true]の場合、一致します.
	 */
	public static final boolean eqEng(String src, int start, int end,
			String dest) {
		return AZ_az.eq(src, start, end - start, dest);
	}

	/**
	 * 英字の大文字小文字を区別しない、文字indexOf.
	 * 
	 * @param buf
	 *            設定対象の文字情報を設定します.
	 * @param chk
	 *            チェック対象の文字情報を設定します.
	 * @param off
	 *            設定対象のオフセット値を設定します.
	 * @return int マッチする位置が返却されます. [-1]の場合は情報は存在しません.
	 */
	public static final int indexOfEng(final String buf, final String chk) {
		return AZ_az.indexOf(buf, chk, 0);
	}

	/**
	 * 英字の大文字小文字を区別しない、文字indexOf.
	 * 
	 * @param buf
	 *            設定対象の文字情報を設定します.
	 * @param chk
	 *            チェック対象の文字情報を設定します.
	 * @param off
	 *            設定対象のオフセット値を設定します.
	 * @return int マッチする位置が返却されます. [-1]の場合は情報は存在しません.
	 */
	public static final int indexOfEng(final String buf, final String chk,
			final int off) {
		return AZ_az.indexOf(buf, chk, off);
	}

	/**
	 * 小文字変換.
	 * 
	 * @param s
	 *            対象の文字列を設定します.
	 * @return String 変換された情報が返却されます.
	 */
	public static final String toLowerCase(String s) {
		return AZ_az.toLowerCase(s);
	}

	/**
	 * 小文字変換.
	 * 
	 * @param s
	 *            対象のcharを設定します.
	 * @return char 変換された情報が返却されます.
	 */
	public static final char toLowerCase(char c) {
		return AZ_az._mM[c];
	}

	/**
	 * 文字情報の置き換え.
	 * 
	 * @param src
	 *            置き換え元の文字列を設定します.
	 * @param s
	 *            置き換え文字条件を設定します.
	 * @param d
	 *            置き換え先の文字条件を設定します.
	 * @return String 文字列が返却されます.
	 */
	public static final String changeString(String src, String s, String d) {
		return changeString(src, 0, src.length(), s, d);
	}

	/**
	 * 文字情報の置き換え.
	 * 
	 * @param src
	 *            置き換え元の文字列を設定します.
	 * @param off
	 *            置き換え元文字のオフセット値を設定します.
	 * @param len
	 *            置き換え元文字の長さを設定します.
	 * @param s
	 *            置き換え文字条件を設定します.
	 * @param d
	 *            置き換え先の文字条件を設定します.
	 * @return String 文字列が返却されます.
	 */
	public static final String changeString(String src, int off, int len,
			String s, String d) {
		int j, k;
		char t = s.charAt(0);
		int lenS = s.length();
		StringBuilder buf = new StringBuilder(len);
		for (int i = off; i < len; i++) {
			if (src.charAt(i) == t) {
				j = i;
				k = 0;
				while (++k < lenS && ++j < len && src.charAt(j) == s.charAt(k))
					;
				if (k >= lenS) {
					buf.append(d);
					i += (lenS - 1);
				} else {
					buf.append(t);
				}
			} else {
				buf.append(src.charAt(i));
			}
		}
		return buf.toString();
	}

	/**
	 * 指定文字内のコーテーションインデントを1つ上げる.
	 * 
	 * @param string
	 *            対象の文字列を設定します.
	 * @param indent
	 *            対象のインデント値を設定します. 0を設定した場合は１つインデントを増やします。
	 *            -1を設定した場合は１つインデントを減らします。
	 * @param dc
	 *            [true]の場合、ダブルコーテーションで処理します.
	 * @return String 変換された文字列が返されます.
	 */
	public static final String indentCote(String string, int indent, boolean dc) {
		if (string == null || string.length() <= 0) {
			return string;
		}
		char cote = (dc) ? '\"' : '\'';
		int len = string.length();
		char c;
		int j;
		int yenLen = 0;
		StringBuilder buf = new StringBuilder((int) (len * 1.25d));
		for (int i = 0; i < len; i++) {
			if ((c = string.charAt(i)) == cote) {
				if (yenLen > 0) {
					if (indent == -1) {
						yenLen >>= 1;
					} else {
						yenLen <<= 1;
					}
					for (j = 0; j < yenLen; j++) {
						buf.append("\\");
					}
					yenLen = 0;
				}
				if (indent == -1) {
					buf.append(cote);
				} else {
					buf.append("\\").append(cote);
				}
			} else if ('\\' == c) {
				yenLen++;
			} else {
				if (yenLen != 0) {
					for (j = 0; j < yenLen; j++) {
						buf.append("\\");
					}
					yenLen = 0;
				}
				buf.append(c);
			}
		}
		if (yenLen != 0) {
			for (j = 0; j < yenLen; j++) {
				buf.append("\\");
			}
		}
		return buf.toString();
	}

	/**
	 * 指定文字内のダブルコーテーションインデントを1つ上げる.
	 * 
	 * @param string
	 *            対象の文字列を設定します.
	 * @return String 変換された文字列が返されます.
	 */
	public static final String upIndentDoubleCote(String string) {
		return indentCote(string, 0, true);
	}

	/**
	 * 指定文字内のシングルコーテーションインデントを1つ上げる.
	 * 
	 * @param string
	 *            対象の文字列を設定します.
	 * @return String 変換された文字列が返されます.
	 */
	public static final String upIndentSingleCote(String string) {
		return indentCote(string, 0, false);
	}

	/**
	 * 指定文字内のダブルコーテーションインデントを1つ下げる.
	 * 
	 * @param string
	 *            対象の文字列を設定します.
	 * @return String 変換された文字列が返されます.
	 */
	public static final String downIndentDoubleCote(String string) {
		// 文字列で検出されるダブルコーテーションが￥始まりの場合は、処理する.
		boolean exec = false;
		int len = string.length();
		char c, b;
		b = 0;
		for (int i = 0; i < len; i++) {
			c = string.charAt(i);
			if (c == '\"') {
				if (b == '\\') {
					exec = true;
				}
				break;
			}
			b = c;
		}
		if (exec) {
			return indentCote(string, -1, true);
		}
		return string;
	}

	/**
	 * 指定文字内のシングルコーテーションインデントを1つ下げる.
	 * 
	 * @param string
	 *            対象の文字列を設定します.
	 * @return String 変換された文字列が返されます.
	 */
	public static final String downIndentSingleCote(String string) {
		// 文字列で検出されるシングルコーテーションが￥始まりの場合は、処理する.
		boolean exec = false;
		int len = string.length();
		char c, b;
		b = 0;
		for (int i = 0; i < len; i++) {
			c = string.charAt(i);
			if (c == '\'') {
				if (b == '\\') {
					exec = true;
				}
				break;
			}
			b = c;
		}
		if (exec) {
			return indentCote(string, -1, false);
		}
		return string;
	}

}
