package org.maachang.rimdb.table;

import org.maachang.rimdb.Table;

/**
 * テーブル.
 * 
 * @version 2014/07/07
 * @author masahito suzuki
 * @since rimdb-1.00
 */
public final class TableImpl implements Table {

	protected final Columns columns;
	protected final Uniques uniques;
	protected final Indexs indexs;
	protected final RowImpl row;
	protected final String name;
	protected final int length;

	/**
	 * コンストラクタ.
	 * 
	 * @param columns
	 *            対象のカラムリストを設定します.
	 * @param Uniques
	 *            ユニークカラムリストを設定します.
	 * @param name
	 *            テーブル名を設定します.
	 * @param length
	 *            データ行数を設定します.
	 */
	public TableImpl(Columns columns, Uniques uniques, Indexs indexs,
			String name, int length) {
		this.columns = columns;
		this.uniques = uniques;
		this.indexs = indexs;
		if (indexs != null) {
			this.row = new RowImpl(columns, indexs.positionColumns);
		} else {
			this.row = new RowImpl(columns, null);
		}
		this.name = name;
		this.length = length;
	}

	/**
	 * テーブル利用後のリセット. この処理は、Rowデータなど、アクセスが終了した後に呼び出す必要があります.
	 */
	public final void reset() {
		row.clear();
	}

	/**
	 * テーブル名を取得.
	 * 
	 * @return String テーブル名が返却されます.
	 */
	public final String getName() {
		return name;
	}

	/**
	 * 全行数を取得.
	 * 
	 * @return int 行数が返却されます.
	 */
	public final int length() {
		return length;
	}

	/**
	 * Columnsオブジェクトを取得.
	 * 
	 * @return Columns Columnsオブジェクトが返却されます.
	 */
	public final Columns getColumns() {
		return columns;
	}

	/**
	 * Uniquesオブジェクトを取得.
	 * 
	 * @return Uniques Uniquesオブジェクトが返却されます.
	 */
	public final Uniques getUniques() {
		return uniques;
	}

	/**
	 * Indexsオブジェクトを取得.
	 * 
	 * @return Indexs Indexsオブジェクトが返却されます.
	 */
	public final Indexs getIndexs() {
		return indexs;
	}

	/**
	 * カラム数を取得.
	 * 
	 * @return int カラム数が返却されます.
	 */
	public final int getColumnLength() {
		return columns.length;
	}

	/**
	 * カラム名を取得.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @return String カラム名が返却されます.
	 */
	public final String getColumnName(final int no) {
		return columns.columnNames[no];
	}

	/**
	 * 行オブジェクトを取得.
	 * 
	 * @return Row 行情報が返却されます.
	 */
	public final RowImpl getRow() {
		row.clear();
		return row;
	}

	/**
	 * 行オブジェクトを取得.
	 * 
	 * @param no
	 *            行番号を設定します.
	 * @return Row 行情報が返却されます.
	 */
	public final RowImpl getRow(final int no) {
		row.position(no);
		return row;
	}

	/**
	 * 指定カラム項番がユニーク条件かチェック.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @return boolean [true]の場合、ユニーク条件です.
	 */
	public final boolean isUnique(final int no) {
		return uniques.contains(no);
	}

	/**
	 * 指定カラム項番がユニーク条件かチェック.
	 * 
	 * @param name
	 *            対象のカラム名を設定します.
	 * @return boolean [true]の場合、ユニーク条件です.
	 */
	public final boolean isUnique(final String name) {
		final int p = columns.getColumnNo(name);
		if (p == -1) {
			return false;
		}
		return uniques.contains(p);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return int 検索場所が返却されます.
	 */
	public final int search(final int value) {
		return search(0, value);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return int 検索場所が返却されます.
	 */
	public final int search(final long value) {
		return search(0, value);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param value
	 *            検索条件を設定します.
	 * @return Row 検索位置のRow情報が返却されます.
	 */
	public final RowImpl searchRow(final int value) {
		return searchRow(0, value);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param value
	 *            検索条件を設定します.
	 * @return Row 検索位置のRow情報が返却されます.
	 */
	public final RowImpl searchRow(final long value) {
		return searchRow(0, value);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return int 検索場所が返却されます.
	 */
	public final int search(final int no, final int value) {
		return uniques.search(no, value);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return int 検索場所が返却されます.
	 */
	public final int search(final int no, final long value) {
		return uniques.search(no, value);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return int 検索場所が返却されます.
	 */
	public final int search(final String name, final int value) {
		final int p = columns.getColumnNo(name);
		if (p == -1) {
			return -1;
		}
		return uniques.search(p, value);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return int 検索場所が返却されます.
	 */
	public final int search(final String name, final long value) {
		final int p = columns.getColumnNo(name);
		if (p == -1) {
			return -1;
		}
		return uniques.search(p, value);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return Row 検索位置のRow情報が返却されます.
	 */
	public final RowImpl searchRow(final int no, final int value) {
		final int p = uniques.search(no, value);
		if (p == -1) {
			return null;
		}
		return getRow(p);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return Row 検索位置のRow情報が返却されます.
	 */
	public final RowImpl searchRow(final int no, final long value) {
		final int p = uniques.search(no, value);
		if (p == -1) {
			return null;
		}
		return getRow(p);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return Row 検索位置のRow情報が返却されます.
	 */
	public final RowImpl searchRow(final String name, final int value) {
		int p = columns.getColumnNo(name);
		if (p == -1 || (p = uniques.search(p, value)) == -1) {
			return null;
		}
		return getRow(p);
	}

	/**
	 * ユニーク条件を検索.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param value
	 *            検索条件を設定します.
	 * @return Row 検索位置のRow情報が返却されます.
	 */
	public final RowImpl searchRow(final String name, final long value) {
		int p = columns.getColumnNo(name);
		if (p == -1 || (p = uniques.search(p, value)) == -1) {
			return null;
		}
		return getRow(p);
	}

}
