package org.maachang.rimdb.search;

import java.lang.ref.SoftReference;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 検索オブジェクトをキャッシュ管理.
 * 
 * @version 2014/07/20
 * @author masahito suzuki
 * @since rimdb-1.00
 */
public final class CacheSearch {
	private final Map<String, SoftReference<SearchImpl>> cache = new ConcurrentHashMap<String, SoftReference<SearchImpl>>();

	/**
	 * キャッシュ情報を全部クリア.
	 */
	public final void clear() {
		cache.clear();
	}

	/**
	 * キャッシュ情報を設定.
	 * 
	 * @param name
	 *            キャッシュ名を設定します.
	 * @param search
	 *            キャッシュ対象の検索オブジェクトを設定します.
	 */
	public final void set(final String name, final SearchImpl search) {
		if (name != null && search != null) {
			cache.put(name, new SoftReference<SearchImpl>(search));
		}
	}

	/**
	 * キャッシュ情報を取得.
	 * 
	 * @param name
	 *            キャッシュ名を設定します.
	 * @return Search 検索オブジェクトが返却されます. [null]の場合は存在しないか、キャッシュクリアされています.
	 */
	public final SearchImpl get(final String name) {
		if (name == null) {
			return null;
		}
		SearchImpl ret;
		SoftReference<SearchImpl> c = cache.get(name);
		if (c != null && (ret = c.get()) != null) {
			return ret.copy();
		}
		if (c != null) {
			cache.remove(name);
		}
		return null;
	}

	/**
	 * キャッシュ情報をクリア.
	 * 
	 * @param name
	 *            キャッシュ名を設定します.
	 */
	public final void remove(final String name) {
		cache.remove(name);
	}

}
