package org.maachang.rimdb.index;

import org.maachang.rimdb.util.NAdd;

/**
 * インデックス情報.
 * 
 * @version 2014/06/29
 * @author masahito suzuki
 * @since rimdb-1.00
 */
public abstract class Index<T> {

	/** index情報. **/
	protected int columnNo;
	protected String columnName;
	protected int maxLine;
	protected int length;
	protected int[][] indexToLine;
	protected MaskLine[] maskLine;
	protected LineIndex lineIndex;
	protected SortLines sortLineList;

	/**
	 * コンストラクタ.
	 * 
	 * @param index
	 *            対象のインデックスを設定します.
	 * @param line
	 *            対象の行情報を設定します.
	 * @param max
	 *            全体長を設定します.
	 */
	protected Index(int[][] line, int max) {
		this.maxLine = max;
		this.length = line.length;
		this.indexToLine = line;
		this.maskLine = MaskLine.create(line);
		this.lineIndex = new LineIndex(line, max);
		this.sortLineList = new SortLines(line, max);
	}

	/**
	 * カラム項番、名前を設定.
	 * 
	 * @param no
	 *            対象のカラム項番を設定します.
	 * @param name
	 *            対象のカラム名を設定します.
	 */
	public final void setColumnNoAndName(final int no, final String name) {
		columnNo = no;
		columnName = name;
	}

	/**
	 * カラム項番を取得.
	 * 
	 * @return int カラム項番が返却されます.
	 */
	public final int getColumnNo() {
		return columnNo;
	}

	/**
	 * カラム名を取得.
	 * 
	 * @return String カラム名が返却されます.
	 */
	public final String getColumnName() {
		return columnName;
	}

	/**
	 * 全行数を取得.
	 * 
	 * @return int 全行数が返却されます.
	 */
	public final int getMaxLine() {
		return maxLine;
	}

	/**
	 * インデックス数を取得.
	 * 
	 * @return int インデックス数が返却されます.
	 */
	public final int getLength() {
		return length;
	}

	/**
	 * インデックスに対する行情報群を取得.
	 * 
	 * @return int[][] インデックスに対する行情報群が返却されます.
	 */
	public final int[][] getIndexToLine() {
		return indexToLine;
	}

	/**
	 * 行インデックスマスク群を取得.
	 * 
	 * @return LineMaskIndex[] 行インデックスマスク群が返却されます.
	 */
	public final MaskLine[] getMaskLine() {
		return maskLine;
	}

	/**
	 * 行インデックスマスク群を取得.
	 * 
	 * @param pos
	 *            対象のインデックス位置を設定します.
	 * @return LineMaskIndex 行インデックスマスクが返却されます.
	 */
	public final MaskLine getMaskLine(int pos) {
		return maskLine[pos];
	}

	/**
	 * 行インデックスを取得.
	 * 
	 * @return LineIndex 行インデックスが返却されます.
	 */
	public final LineIndex getLineIndex() {
		return lineIndex;
	}

	/**
	 * 行に対するソート情報を取得.
	 * 
	 * @return SortLines 行に対するソート情報が返却されます.
	 */
	public final SortLines getSortLines() {
		return sortLineList;
	}

	/**
	 * 一致検索.
	 * 
	 * @param key
	 *            対象のキーを設定します.
	 * @return int インデックス番号が返却されます.
	 */
	public abstract int eq(T key);

	/**
	 * 大なり検索.
	 * 
	 * @param eq
	 *            一致を許可する場合には[true]をセットします.
	 * @param key
	 *            対象のキーを設定します.
	 * @return int インデックス番号が返却されます.
	 */
	public abstract int big(boolean eq, T key);

	/**
	 * 小なり検索.
	 * 
	 * @param eq
	 *            一致を許可する場合には[true]をセットします.
	 * @param key
	 *            対象のキーを設定します.
	 * @return int インデックス番号が返却されます.
	 */
	public abstract int small(boolean eq, T key);

	/**
	 * between検索.
	 * 
	 * @param out
	 *            格納行情報を設定します.
	 * @param start
	 *            開始値を設定します.
	 * @param end
	 *            終了値を設定します.
	 * @return boolean [true]の場合、範囲取得に成功しました.
	 */
	public abstract boolean between(int[] out, T start, T end);

	/**
	 * 部分一致検索.
	 * 
	 * @param result
	 *            一致インデックス位置を格納するオブジェクトを設定します.
	 * @param key
	 *            対象のLike条件を設定します.
	 */
	public abstract void like(NAdd result, String key);
}
