package org.maachang.mimdb.core.impl ;

import org.maachang.mimdb.MimdbException;


/**
 * Andフラグリスト.
 * 
 * @version 2013/10/27
 * @author masahito suzuki
 * @since MasterInMemDB 1.00
 */
public final class FlagsAnd extends Flags {
    
    /** And元フラグ. **/
    private int[][] src ;
    
    /** andのmin,max条件. **/
    private int minAndPos = -2 ;
    private int maxAndPos = -2 ;
    
    /**
     * コンストラクタ.
     */
    public FlagsAnd() {
        
    }
    
    /**
     * コンストラクタ.
     * @param f And元のオブジェクトをセットします.
     *          渡されたオブジェクトは内部でクリアされます.
     */
    public FlagsAnd( Flags f ) {
        create( f ) ;
    }
    
    /** デストラクタ. **/
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * 情報生成.
     * @param f And元のオブジェクトをセットします.
     *          渡されたオブジェクトは内部でクリアされます.
     */
    public final void create( Flags f ) {
        int[][] s = f.flags ;
        int m = f.max ;
        f.destroy() ;
        
        src = s ;
        max = m ;
        flags = new int[ src.length ][] ;
        minAndPos = -2 ;
        maxAndPos = -2 ;
    }
    
    /**
     * 情報クリア.
     */
    public final void destroy() {
        super.destroy() ;
        src = null ;
        minAndPos = -2 ;
        maxAndPos = -2 ;
    }
    
    /**
     * Andモード取得.
     * @return boolean [true]の場合、ANDモードです.
     */
    public final boolean isAnd() {
        return true ;
    }
    
    /**
     * 指定位置のフラグをON.
     * @param no 対象の項番を設定します.
     */
    public final void add( final int no ) {
        int[] ff ;
        final int n = no >> 10 ;
        final int nn = no & 0x3ff ;
        
        // AND対象位置の条件が0の場合は処理しない.
        if( ( ff = src[ n ] ) != null &&
            ( ff[ nn >> 5 ] & ( 1 << ( nn & 0x1f ) ) ) != 0 ) {
            
            // ブロック位置条件を取得.
            if( ( ff = flags[ n ] ) == null ) {
                ff = new int[ BLOCK_INNER_SIZE ] ;
                flags[ n ] = ff ;
            }
            
            // フラグセット.
            ff[ nn >> 5 ] |= ( 1 << ( nn & 0x1f ) ) ;
            
        }
    }
    
    /**
     * 指定位置のフラグをON.
     * @param array 対象の項番群を設定します.
     */
    public final void addArray( final int[] array ) {
        int a,n,nn ;
        int[] ff ;
        
        final int len = array.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            
            a = array[ i ] ;
            n = a >> 10 ;
            nn = a & 0x3ff ;
            
            // AND対象位置の条件が0の場合は処理しない.
            if( ( ff = src[ n ] ) != null &&
                ( ff[ nn >> 5 ] & ( 1 << ( nn & 0x1f ) ) ) != 0 ) {
                
                // ブロック位置条件を取得.
                if( ( ff = flags[ n ] ) == null ) {
                    ff = new int[ BLOCK_INNER_SIZE ] ;
                    flags[ n ] = ff ;
                }
                
                // フラグセット.
                ff[ nn >> 5 ] |= ( 1 << ( nn & 0x1f ) ) ;
                
            }
        }
    }
    
    /**
     * 指定位置のフラグを設定.
     * @param no 対象の項番を設定します.
     * @param f フラグ条件を設定します.
     *          [0]がOFF、[1]がONです.
     */
    public final void set( final int no,final int f ) {
        int[] ff ;
        final int n = no >> 10 ;
        final int nn = no & 0x3ff ;
        
        // AND対象位置の条件が0の場合は処理しない.
        if( ( ff = src[ n ] ) != null &&
            ( ff[ nn >> 5 ] & ( 1 << ( nn & 0x1f ) ) ) != 0 ) {
            
            // ブロック位置条件を取得.
            if( ( ff = flags[ n ] ) == null ) {
                ff = new int[ BLOCK_INNER_SIZE ] ;
                flags[ n ] = ff ;
            }
            
            // フラグセット.
            ff[ nn >> 5 ] = ( ff[ nn >> 5 ] & ~( 1 << ( nn & 0x1f ) ) ) |
                ( ( f & 0x1 ) << ( nn & 0x1f ) ) ;
            
        }
    }
    
    /**
     * 指定位置のフラグを設定.
     * @param array 対象の項番群を設定します.
     * @param f フラグ条件を設定します.
     *          [0]がOFF、[1]がONです.
     */
    public final void setArray( final int[] array,final int f ) {
        int a,n,nn ;
        int[] ff ;
        
        final int len = array.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            
            a = array[ i ] ;
            n = a >> 10 ;
            nn = a & 0x3ff ;
            
            // AND対象位置の条件が0の場合は処理しない.
            if( ( ff = src[ n ] ) != null &&
                ( ff[ nn >> 5 ] & ( 1 << ( nn & 0x1f ) ) ) != 0 ) {
                
                // ブロック位置条件を取得.
                if( ( ff = flags[ n ] ) == null ) {
                    ff = new int[ BLOCK_INNER_SIZE ] ;
                    flags[ n ] = ff ;
                }
                
                // フラグセット.
                ff[ nn >> 5 ] = ( ff[ nn >> 5 ] & ~( 1 << ( nn & 0x1f ) ) ) |
                    ( ( f & 0x1 ) << ( nn & 0x1f ) ) ;
                
            }
        }
    }
    
    /**
     * 全てのフラグをONに設定.
     */
    public final void all() {
        int[] ff,sf ;
        int len = flags.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            
            // AND条件の情報が存在する場合.
            if( ( sf = src[ i ] ) != null ) {
                
                // ブロック位置条件を取得.
                if( ( ff = flags[ i ] ) == null ) {
                    ff = new int[ BLOCK_INNER_SIZE ] ;
                    flags[ i ] = ff ;
                }
                
                // and条件をコピー.
                System.arraycopy( sf,0,ff,0,BLOCK_INNER_SIZE ) ;
            }
        }
    }
    
    /**
     * 対象の条件をマージ.
     * @param f マージ対象のオブジェクトを設定します.
     */
    public final void marge( Flags f ) {
        if( f.max() != max ) {
            throw new MimdbException( "長さが不一致:" + f.max + " " + max ) ;
        }
        // Flags内のBlockFlags条件を取得.
        int j ;
        int[] ff,bf,sf ;
        int[][] bff = f.flags ;
        final int len = flags.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            
            // AND条件の情報が存在する場合.
            // マージ対象のBlockObjectが存在する場合.
            if( ( sf = src[ i ] ) != null && ( bf = bff[ i ] ) != null ) {
                
                // 設定対象のBlockObjectが存在しない場合は作成.
                if( ( ff = flags[ i ] ) == null ) {
                    ff = new int[ BLOCK_INNER_SIZE ] ;
                    flags[ i ] = ff ;
                }
                
                // 対象の条件をORマージ.
                for( j = 0 ; j < BLOCK_INNER_SIZE ; j ++ ) {
                    ff[ j ] |= ( bf[ j ] & sf[ j ] ) ;
                }
            }
        }
    }
    
    /**
     * AND条件の最小ON位置を取得.
     * ※この情報はキャッシュ化されるので、取り消す場合は
     *   clearAndPositionを呼び出す必要があります.
     * @return int AND条件での最小ON位置が返却されます.
     */
    public int minAndPosition() {
        if( minAndPos == -2 ) {
            minAndPos = minPosition( src ) ;
        }
        return minAndPos ;
    }
    
    /**
     * AND条件の最大ON位置を取得.
     * ※この情報はキャッシュ化されるので、取り消す場合は
     *   clearAndPositionを呼び出す必要があります.
     * @return int AND条件での最大ON位置が返却されます.
     */
    public int maxAndPosition() {
        if( maxAndPos == -2 ) {
            maxAndPos = maxPosition( src ) ;
        }
        return maxAndPos ;
    }
    
    /**
     * AND条件の最小、最大位置キャッシュをクリア.
     */
    public void clearAndPosition() {
        minAndPos = -2 ;
        maxAndPos = -2 ;
    }
}
