package org.maachang.mimdb.core ;

import org.maachang.mimdb.MimdbException;
import org.maachang.mimdb.core.util.Config;

/**
 * 対象ConfファイルからCSVファイル生成.
 * 
 * @version 2013/10/28
 * @author masahito suzuki
 * @since MasterInMemDB 1.00
 */
public final class MimdbTableInstance {
    private MimdbTableInstance() {}
    
    /**
     * Confファイルを設定して、テーブル定義.
     * @param baseFolder 基本フォルダーを設定します.
     * @param name 対象のConfファイル名を設定します.
     * @return String 作成が成功したテーブル名が返却されます.
     * @exception Exception 例外.
     */
    public static final String create( String baseFolder,String name ) throws Exception {
        return create( baseFolder,Config.read( name ) ) ;
    }
    
    /**
     * Confオブジェクトを設定して、テーブル定義.
     * @param baseFolder 基本フォルダーを設定します.
     * @param conf 対象のConfオブジェクトを定義します.
     * @return String 作成が成功したテーブル名が返却されます.
     * @exception Exception 例外.
     */
    public static final String create( String baseFolder,Config conf ) throws Exception {
        // confチェック.
        if( conf == null || !conf.isSection( "mimdb" ) ) {
            throw new MimdbException( "指定Confオブジェクトは、mimdb定義では有りません" ) ;
        }
        // jdbc条件の場合.
        if( conf.isSection( "jdbc" ) ) {
            return MimdbJdbcTable.create( conf ) ;
        }
        // csv条件の場合.
        else if( conf.isSection( "csv" ) && conf.isSection( "column" ) ) {
            return MimdbCsvTable.create( baseFolder,conf ) ;
        }
        else {
            throw new MimdbException( "指定Confオブジェクトは、mimdbテーブル定義では有りません" ) ;
        }
    }
    
    /**
     * DBテーブルIDを取得.
     * @param name 対象のテーブル名を設定します.
     * @return long テーブルIDが返却されます.
     */
    protected static final long begin( String name ) {
        // ロック処理開始.
        MimdbTableManager m = MimdbTableManager.getInstance() ;
        m.begin( name,true ) ;
        return m.updateId( name ) ;
    }
    
    /**
     * テーブル登録完了.
     * ※この処理はbegin処理を呼び出した後、終了処理に対して、必ず呼び出す必要があります.
     * @param name 対象のテーブル名を設定します.
     */
    protected static final void end( String name ) {
        MimdbTableManager.getInstance().end( name ) ;
    }
    
    /**
     * テーブル登録完了.
     * ※この処理はbegin処理を呼び出した後、終了処理に対して、必ず呼び出す必要があります.
     * @param name 対象のテーブル名を設定します.
     */
    protected static final void errorEnd( String name,long id ) {
        MimdbTableManager m = MimdbTableManager.getInstance() ;
        m.rollbackId( name,id ) ;
        m.end( name ) ;
        
    }
    
    /** テーブル名を取得. **/
    protected static final String getTableName( Config conf ) throws Exception {
        String tableName = conf.getString( "mimdb","table",0 ) ;
        if( tableName == null || ( tableName = tableName.trim() ).length() <= 0 ) {
            throw new MimdbException( "テーブル名は定義されてません" ) ;
        }
        return tableName.toLowerCase() ;
    }
    
    /** プライマリーキーを取得. **/
    protected static final String getPrimaryKey( Config conf ) throws Exception {
        String primary = conf.getString( "mimdb","primary",0 ) ;
        if( primary == null || ( primary = primary.trim() ).length() <= 0 ) {
            return null ;
        }
        return primary.toLowerCase() ;
    }
    
    /** テーブル圧縮モードを取得. **/
    protected static final boolean getCompressFlag( Config conf ) throws Exception {
        return conf.getBoolean( "mimdb","compress",0 ) ;
    }
    
    /** テーブル圧縮での１度の行圧縮数を取得. **/
    protected static final int getCompressLength( Config conf ) throws Exception {
        return conf.getInt( "mimdb","compressLength",0 ) ;
    }
    
    /** インデックスカラム名一覧を取得 **/
    protected static final String[] getIndexNames( Config conf ) throws Exception {
        int len = conf.size( "mimdb","index" ) ;
        if( len <= 0 ) {
            return null ;
        }
        String[] ret = new String[ len ] ;
        for( int i = 0 ; i < len ; i ++ ) {
            ret[ i ] = conf.getString( "mimdb","index",i ).trim().toLowerCase() ;
        }
        return ret ;
    }
    
    /** NGramインデックスカラム名一覧を取得 **/
    protected static final String[] getNGramIndexNames( Config conf ) throws Exception {
        int len = conf.size( "mimdb","ngram" ) ;
        if( len <= 0 ) {
            return null ;
        }
        String[] ret = new String[ len ] ;
        for( int i = 0 ; i < len ; i ++ ) {
            ret[ i ] = conf.getString( "mimdb","ngram",i ).trim().toLowerCase() ;
        }
        return ret ;
    }
    
}
