package mirrg.swing.helium.property;

import java.awt.CardLayout;
import java.awt.Frame;
import java.util.function.Supplier;

import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableModel;

import mirrg.struct.hydrogen.Tuple;
import mirrg.swing.helium.DialogMirrg;
import mirrg.swing.helium.logging.HLog;

public class DialogProperties extends DialogMirrg
{

	private Supplier<ManagerProperty> supplierManagerProperty;
	private JScrollPane scrollPane;
	private DefaultTableModel tableModel;

	public DialogProperties(Frame frame, Supplier<ManagerProperty> supplierManagerProperty)
	{
		super(frame, "プロパティマネージャ");
		this.supplierManagerProperty = supplierManagerProperty;

		{
			JMenuBar menuBar = new JMenuBar();
			setJMenuBar(menuBar);

			{
				JMenu menu = new JMenu("メニュー(M)");
				menuBar.add(menu);

				menu.setMnemonic('M');

				{
					JMenuItem menuItem = new JMenuItem("再読み込み(R)");
					menu.add(menuItem);

					menuItem.setMnemonic('R');
					menuItem.addActionListener(e -> {
						refreshAll();
					});
				}

			}

		}

		{
			scrollPane = new JScrollPane();
			add(scrollPane);

			refreshAll();
			onTick.add(this::refresh);

			setLayout(new CardLayout());
		}

		enabledTick = true;
		msTick = 100;

		setDefaultCloseOperation(HIDE_ON_CLOSE);
		pack();
		setLocationByPlatform(true);
	}

	private ManagerProperty getManagerProperty()
	{
		return supplierManagerProperty.get();
	}

	private void refresh()
	{
		if (tableModel == null) return;

		for (int row = 0; row < tableModel.getRowCount(); row++) {

			String id = (String) tableModel.getValueAt(row, 0);
			Property<?> property = getManagerProperty().getProperty(id).orElse(null);

			if (property != null) {
				tableModel.setValueAt(property.getString(), row, 2);
			} else {
				HLog.processExceptionWarning(new RuntimeException("found property id: " + id));
			}

		}

		scrollPane.repaint();
	}

	private void refreshAll()
	{
		Tuple<JTable, DefaultTableModel> res = createTable();
		scrollPane.setViewportView(res.getX());
		tableModel = res.getY();

		scrollPane.repaint();
	}

	private Tuple<JTable, DefaultTableModel> createTable()
	{

		// 添え字2の列だけ編集可能
		DefaultTableModel tableModel = new DefaultTableModel(0, 4) {

			@Override
			public Class<?> getColumnClass(int column)
			{
				return String.class;
			}

			@Override
			public boolean isCellEditable(int row, int column)
			{
				return column == 2;
			}

		};

		// 列ラベル
		tableModel.setColumnIdentifiers(new Object[] {
			"ID",
			"名称",
			"値",
			"型",
		});

		// 全てのプロパティの書き出し
		getManagerProperty().getProperties().forEach(property -> {
			tableModel.addRow(new Object[] {
				property.id,
				property.name,
				property.getString(),
				property.getType(),
			});
		});

		// テーブル編集ハンドル
		tableModel.addTableModelListener(new TableModelListener() {

			@Override
			public void tableChanged(TableModelEvent e)
			{
				for (int row = e.getFirstRow(); row <= e.getLastRow(); row++) {

					if (row == TableModelEvent.HEADER_ROW) {
						continue;
					}

					String id = (String) tableModel.getValueAt(row, 0);
					Property<?> property = getManagerProperty().getProperty(id).orElse(null);
					String string = (String) tableModel.getValueAt(row, 2);

					if (property != null) {
						property.setString(string);
					} else {
						HLog.processExceptionWarning(new RuntimeException("found property id: " + id));
					}

				}
			}

		});

		// テーブル生成
		return new Tuple<>(new JTable(tableModel), tableModel);
	}

}
