package mirrg.simulation.cart.almandine.factory;

import java.awt.Graphics2D;
import java.awt.Point;
import java.util.ArrayList;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.Hashtable;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import mirrg.simulation.cart.almandine.GameAlmandine;
import mirrg.simulation.cart.almandine.IFrameGameAlmandine;
import mirrg.swing.helium.logging.HLog;

import com.thoughtworks.xstream.annotations.XStreamOmitField;

/**
 * ファクトリ直下にはプライマリエンティティしか追加できない。
 * このクラスはXStream互換です。
 */
public class Factory
{

	/**
	 * {@link #hookEditPrimaries(Runnable)}内でのみ、構成と固有IDを変更してよい。
	 */
	public ArrayList<Primary> primaries = new ArrayList<>();

	public void onLoad(GameAlmandine game)
	{
		primaries.forEach(primary -> primary.setFactoryAll(game));
		processEditPrimaries();
	}

	@XStreamOmitField
	private Hashtable<Integer, Entity> hashIdToEntity;

	public void tick(double deltaSecond)
	{
		try {
			primaries.forEach(primary -> tick(primary, deltaSecond));
		} catch (ConcurrentModificationException e) {
			HLog.processExceptionWarning(e);
		}
	}

	public void processEditPrimaries()
	{

		// フックしたプライマリ変更ハンドラの消化
		getListenersEditPrimaries().forEach(Runnable::run);
		getListenersEditPrimaries().clear();

		cacheEntityIds();

	}

	private synchronized void cacheEntityIds()
	{

		// IDのキャッシュ
		// 再び呼び出されるまでID分布が変更されることはない。
		hashIdToEntity = new Hashtable<>();
		getEntities().forEach(entity -> hashIdToEntity.put(entity.getId(), entity));

	}

	private void tick(Primary primary, double deltaSecond)
	{
		try {
			primary.tick(deltaSecond);
		} catch (IllegalEntityIdException e) {

			// 破綻プライマリの削除予約
			hookEditPrimaries(() -> {
				primaries.remove(primary);
				HLog.processException(e,
					String.format(
						"接続エンティティの末端が削除されました。接続エンティティを削除します: id=%s, class=%s",
						primary.getId(),
						primary.getClass().getSimpleName()), false);
			});

		}
	}

	// TODO
	public void tickSuperHighSpeed()
	{
		tick(getNextSuperHighSpeedDuration());
	}

	private double getNextSuperHighSpeedDuration()
	{
		return primaries.stream()
			.mapToDouble(primary -> {
				try {
					return primary.getDurationVoid();
				} catch (Exception e) {
					return 1;
				}
			})
			.min()
			.orElse(1);
	}

	@XStreamOmitField
	private ArrayList<Runnable> listenersEditPrimaries;

	private ArrayList<Runnable> getListenersEditPrimaries()
	{
		if (listenersEditPrimaries == null) listenersEditPrimaries = new ArrayList<>();
		return listenersEditPrimaries;
	}

	/**
	 * このリスナー内からのみ、{@link #primaries}の編集を行ってよい。
	 */
	public void hookEditPrimaries(Runnable runnable)
	{
		getListenersEditPrimaries().add(runnable);
	}

	public void render(Graphics2D graphics, IFrameGameAlmandine frameGameAlmandine)
	{
		frameGameAlmandine.doTranslate(graphics, () -> {

			try {
				primaries.forEach(primary -> draw(primary, graphics));
				primaries.forEach(primary -> drawOverlay(primary, graphics));
			} catch (ConcurrentModificationException e) {
				HLog.processExceptionWarning(e);
			}

		});
	}

	private void draw(Entity entity, Graphics2D graphics)
	{
		try {
			entity.draw(graphics);
		} catch (IllegalEntityIdException e) {
			// 破綻プライマリは描画時は無視する
			//TODO HLog.processExceptionWarning(e);
		}
	}

	private void drawOverlay(Entity entity, Graphics2D graphics)
	{
		try {
			entity.drawOverlay(graphics);
		} catch (IllegalEntityIdException e) {
			// 破綻プライマリは描画時は無視する
		}
	}

	/**
	 * まだ利用されていないエンティティIDをランダムに生成する。
	 */
	public int freeId()
	{
		int id;
		do {
			id = (int) (Math.random() * 100000000);
		} while (getEntity(id).isPresent());
		return id;
	}

	public void addLater(Primary primary)
	{
		hookEditPrimaries(() -> {

			primaries.add(primary);

		});
	}

	public void deleteLater(Primary primary)
	{
		hookEditPrimaries(() -> {

			if (primaries.contains(primary)) {
				primaries.remove(primary);
			} else {
				HLog.error("削除できない型のプライマリオブジェクトです: "
					+ primary + "(" + primary.getClass() + ")");
			}

		});
	}

	/////////////////////////////////////////////////////

	/**
	 * プライマリエンティティ内部の子エンティティを含むすべてのエンティティをZオーダー順に取得する。
	 */
	public Stream<Entity> getEntities()
	{
		return primaries.stream()
			.flatMap(primary -> Stream.of(Stream.of(primary), primary.getChildren().map(part -> part))
				.flatMap(stream -> stream));
	}

	/**
	 * 全てのプライマリエンティティをZオーダー順に取得する。
	 */
	@Deprecated
	public Stream<Primary> getPrimaries()
	{
		return primaries.stream();
	}

	/**
	 * 特定のIDを持つエンティティを取得する。
	 */
	public synchronized Optional<Entity> getEntity(int id)
	{
		return Optional.ofNullable(hashIdToEntity.get(id));
	}

	/**
	 * 特定のIDを持つエンティティを取得する。
	 */
	public synchronized Entity getEntityOrThrow(int id) throws IllegalEntityIdException
	{
		Entity entity = hashIdToEntity.get(id);
		if (entity == null) {
			throw new IllegalEntityIdException(id);
		}
		return entity;
	}

	/**
	 * 特定のIDを持つエンティティを指定のクラスにキャストを試行して取得する。
	 */
	@SuppressWarnings("unchecked")
	public <T extends Entity> Optional<T> getEntity(int id, Class<T> classSuper)
	{
		Optional<Entity> optionalEntity = getEntity(id);
		if (!optionalEntity.isPresent()) return Optional.empty();
		Entity entity = optionalEntity.get();
		if (!classSuper.isInstance(entity)) return Optional.empty();
		return Optional.of((T) entity);
	}

	/**
	 * 特定のIDを持つエンティティを指定のクラスにキャストを試行して取得する。
	 */
	@SuppressWarnings("unchecked")
	public <T extends Entity> T getEntityOrThrow(int id, Class<T> classSuper) throws IllegalEntityIdException
	{
		Entity entity = getEntityOrThrow(id);
		if (!classSuper.isInstance(entity)) throw new IllegalEntityIdException(id, entity.getClass(), classSuper);
		return (T) entity;
	}

	/////////////////////////////////////////////////////

	/**
	 * ストリームから特定のクラスのエンティティを選択する。
	 */
	@SuppressWarnings("unchecked")
	public static <T extends Entity, C extends T> Stream<C> filterClass(Stream<T> stream, Class<C> classSuper)
	{
		return stream
			.filter(entity -> classSuper.isInstance(entity))
			.map(entity -> (C) entity);
	}

	/**
	 * ストリームから特定の位置を含むエンティティを選択する。
	 * 表示順にソートする（登録順とは逆順）。
	 */
	public static <T extends Entity> ArrayList<T> filterHover(Stream<T> stream, int x, int y)
	{
		ArrayList<T> list = stream
			.filter(entity -> {
				try {
					return entity.isHover(x, y);
				} catch (IllegalEntityIdException e) {
					return false;
				}
			})
			.collect(Collectors.toCollection(ArrayList::new));
		Collections.reverse(list);
		return list;
	}

	/**
	 * ストリームから特定の位置を含むエンティティを選択する。
	 * 表示順にソートする（登録順とは逆順）。
	 */
	public static <T extends Entity> ArrayList<T> filterHover(Stream<T> stream, Point point)
	{
		return filterHover(stream, point.x, point.y);
	}

	/**
	 * ストリームから選択中のプライマリエンティティを選択する。
	 */
	public static <T extends Primary> Stream<T> filterSelected(Stream<T> stream)
	{
		return stream
			.filter(primary -> primary.selected);
	}

	/////////////////////////////////////////////////////

	/**
	 * 全てのプライマリエンティティの選択を解除する。
	 */
	public void resetSelection()
	{
		getPrimaries()
			.forEach(primary -> primary.selected = false);
	}

	/**
	 * 指定パーツの親プライマリエンティティを返す。
	 */
	public Optional<Primary> getParent(Part part)
	{
		return getPrimaries()
			.filter(primary -> primary.getChildren().anyMatch(part2 -> part2 == part))
			.findFirst();
	}

}
