/*
 * (c) Copyright Sysdeo SA 2001, 2002.
 * All Rights Reserved.
 */

package com.sysdeo.eclipse.tomcat.editors;

import static com.sysdeo.eclipse.tomcat.TomcatPluginResources.PREF_PAGE_SELECTALL_LABEL;
import static com.sysdeo.eclipse.tomcat.TomcatPluginResources.PREF_PAGE_UNSELECTALL_LABEL;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.CheckedListDialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.DialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.LayoutUtil;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.ide.IDE;

import com.sysdeo.eclipse.tomcat.TomcatLauncherPlugin;

/**
 * ProjectListEditor
 *
 */
public class ProjectListEditor {

	/** fProjectsList */
	private CheckedListDialogField<ProjectListElement> fProjectsList;
	/** fExcludedNatures */
	private String[] fExcludedNatures;

	/**
	 * Constructor
	 */
	public ProjectListEditor() {
		this(new String[0]);
	}

	/**
	 * Constructor
	 * @param excludedNatures String[]
	 */
	public ProjectListEditor(final String[] excludedNatures) {
		this.fExcludedNatures = excludedNatures.clone();
		String[] buttonLabels = new String[] {
			PREF_PAGE_SELECTALL_LABEL,
			PREF_PAGE_UNSELECTALL_LABEL
		};

		this.fProjectsList = new CheckedListDialogField<>(null, buttonLabels, new MyLabelProvider());
		this.fProjectsList.setCheckAllButtonIndex(0);
		this.fProjectsList.setUncheckAllButtonIndex(1);
		updateProjectsList();
	}

	/**
	 *
	 * @param enabled boolean
	 */
	public void setEnabled(final boolean enabled) {
		this.fProjectsList.setEnabled(enabled);
	}

	/**
	 *
	 * @param jproject IJavaProject
	 */
	public void init(final IJavaProject jproject) {
		if (jproject != null) {
			updateProjectsList();
		}
	}

	/**
	 *
	 * @param label Strng
	 */
	public void setLabel(final String label) {
		this.fProjectsList.setLabelText(label);
	}

	/**
	 * updateProjectsList
	 */
	private void updateProjectsList() {
		try {
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();

			List<ProjectListElement> projectsList = new ArrayList<>();
			for (final IProject project : root.getProjects()) {
				if (project.isOpen()) {
					boolean accept = true;
					for (final String nature : this.fExcludedNatures) {
						if (project.hasNature(nature)) {
							accept = false;
						}
					}
					if (accept) {
						projectsList.add(new ProjectListElement(project));
					}
				}
			}

			this.fProjectsList.setElements(projectsList);
			this.fProjectsList.setCheckedElements(TomcatLauncherPlugin.getProjectsInCP());

		} catch (final CoreException e) {
			this.fProjectsList.setElements(new ArrayList<ProjectListElement>(5));
			e.printStackTrace();
		}
	}

	// -------- UI creation ---------

	/**
	 *
	 * @param parent Composite
	 * @return Control
	 */
	public Control getControl(final Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		LayoutUtil.doDefaultLayout(composite, new DialogField[]{this.fProjectsList}, true, 0, 0);
		return composite;
	}

	/**
	 *
	 * @return List<ProjectListElement>
	 */
	public List<ProjectListElement> getCheckedElements() {
		return this.fProjectsList.getCheckedElements();
	}

	/**
	 *
	 * @param projects List<ProjectListElement>
	 */
	public void setCheckedElements(final List<ProjectListElement> projects) {
		this.fProjectsList.setCheckedElements(projects);
	}

	/**
	 * MyLabelProvider
	 *
	 */
	static class MyLabelProvider extends LabelProvider {
		/**
		 * @see org.eclipse.jface.viewers.LabelProvider#getImage(java.lang.Object)
		 */
		@Override
		public Image getImage(final Object element) {
			IWorkbench workbench = JavaPlugin.getDefault().getWorkbench();
			return workbench.getSharedImages().getImage(IDE.SharedImages.IMG_OBJ_PROJECT);
		}
	}
}
