package batch.status;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;

import common.db.jdbc.Jdbc;
import common.sql.QueryUtil;
import core.exception.PhysicalException;
import core.exception.ThrowableUtil;

/**
 * ジョブ詳細管理テーブル取得／更新クラス
 *
 * @author Tadashi Nakayama
 * @version 1.0.0
 */
public final class JobDetailStatusImpl implements JobDetailStatus {

	/**
	 * ジョブ詳細管理状態取得
	 *
	 * @param conn コネクション
	 * @param seq ジョブ連番
	 * @param dseq バッチ連番
	 * @return ジョブ詳細管理状態リスト
	 */
	@Override
	public JobDetail getJobDetail(final Connection conn, final long seq, final int dseq) {
		Map<String, Object> map = new HashMap<>();
		map.put("JobSeq", Long.valueOf(seq));
		map.put("BatSeq", Integer.valueOf(dseq));

		String query = QueryUtil.getSqlFromFile("SelectJobDetailStatus", this.getClass());
		try (PreparedStatement psmt = QueryUtil.createStatement(
				query, map, Jdbc.wrap(conn)::readonlyStatement)) {
			JobDetail rec = null;
			try (ResultSet rs = psmt.executeQuery()) {
				if (rs.next()) {
					rec = toJobDetailObject(rs);
				}
			}
			return rec;

		} catch (final SQLException ex) {
			ThrowableUtil.error(ex);
			throw new PhysicalException(ex);
		}
	}

	/**
	 * ジョブ詳細管理状態取得
	 *
	 * @param conn コネクション
	 * @param seq ジョブ連番
	 * @return ジョブ詳細管理状態リスト
	 */
	@Override
	public List<JobDetail> selectJobDetails(final Connection conn, final long seq) {
		// 読込
		String query = QueryUtil.getSqlFromFile("SelectJobDetailStatusList", this.getClass());
		LogManager.getLogger().debug(query);
		LogManager.getLogger().debug(() -> "jobseq=" + seq);

		try (PreparedStatement psmt = QueryUtil.createStatement(
				query, Collections.singletonMap("JobSeq", Long.valueOf(seq)),
				Jdbc.wrap(conn)::readonlyStatement)) {
			List<JobDetail> list = new ArrayList<>();
			try (ResultSet rs = psmt.executeQuery()) {
				while (rs.next()) {
					list.add(toJobDetailObject(rs));
				}
			}
			return list;

		} catch (final SQLException ex) {
			ThrowableUtil.error(ex);
			throw new PhysicalException(ex);
		}
	}

	/**
	 * ジョブ詳細管理更新
	 *
	 * @param conn コネクション
	 * @param seq ジョブ連番
	 * @param dseq バッチ連番
	 * @param msg メッセージ
	 * @param status ステータス
	 * @param time 日時
	 * @return 更新レコード数
	 */
	@Override
	public int updateJobDetail(final Connection conn, final long seq, final int dseq,
					final String msg, final JobState status, final Timestamp time) {
		Map<String, Object> map = new HashMap<>();
		map.put("JobSeq", Long.valueOf(seq));
		map.put("BatSeq", Integer.valueOf(dseq));
		map.put("DateTime", time);
		map.put("Status", Integer.valueOf(status.value()));
		map.put("Message", msg);

		String query = QueryUtil.getSqlFromFile("UpdateJobDetail", this.getClass());
		try (PreparedStatement psmt = QueryUtil.createStatement(
						query, map, Jdbc.wrap(conn)::prepareStatement)) {
			// ジョブ詳細管理テーブル更新
			return psmt.executeUpdate();

		} catch (final SQLException ex) {
			ThrowableUtil.error(ex);
			throw new PhysicalException(ex);
		}
	}

	/**
	 * ジョブ詳細管理新規作成
	 *
	 * @param conn コネクション
	 * @param seq ジョブ連番
	 * @param pid プロセスID
	 * @param name バッチ処理名称
	 * @param time 日時
	 * @return バッチシーケンス番号
	 */
	@Override
	public int insertJobDetail(final Connection conn, final long seq,
					final int pid, final String name, final Timestamp time) {

		int dtlno = 1;
		List<JobDetail> dtllst = selectJobDetails(conn, seq);
		if (!dtllst.isEmpty()) {
			dtlno += dtllst.size();
		}

		Map<String, Object> map = new HashMap<>();
		map.put("JobSeq", Long.valueOf(seq));
		map.put("BatSeq", Integer.valueOf(dtlno));
		map.put("DateTime", time);
		map.put("Status", Integer.valueOf(JobState.ID_B_EDIT.value()));
		map.put("Pid", Integer.valueOf(pid));
		map.put("BatName", name);

		String query = QueryUtil.getSqlFromFile("InsertJobDetail", this.getClass());
		try (PreparedStatement psmt = QueryUtil.createStatement(
				query, map, Jdbc.wrap(conn)::prepareStatement)) {
			// ジョブ管理テーブル更新
			int ret = psmt.executeUpdate();
			if (ret != 1) {
				throw new IllegalStateException(String.valueOf(ret));
			}
			return dtlno;

		} catch (final SQLException ex) {
			ThrowableUtil.error(ex);
			throw new PhysicalException(ex);
		}
	}

	/**
	 * ジョブ詳細管理削除
	 *
	 * @param conn コネクション
	 * @param seq ジョブ連番
	 * @return 削除件数
	 */
	@Override
	public int deleteJobDetail(final Connection conn, final long seq) {

		String query = QueryUtil.getSqlFromFile("DeleteJobDetail", this.getClass());
		try (PreparedStatement psmt = QueryUtil.createStatement(
				query, Collections.singletonMap("JobSeq", Long.valueOf(seq)),
						Jdbc.wrap(conn)::prepareStatement)) {
			return psmt.executeUpdate();
		} catch (final SQLException ex) {
			ThrowableUtil.error(ex);
			throw new PhysicalException(ex);
		}
	}

	/**
	 * ジョブ詳細オブジェクト取得
	 *
	 * @param rs 結果セット
	 * @return ジョブ詳細オブジェクト
	 * @throws SQLException SQL例外
	 */
	private JobDetail toJobDetailObject(final ResultSet rs) throws SQLException {
		JobDetail ret = new JobDetail();
		ret.setJobSeq(rs.getLong("JOB_SEQ"));
		ret.setBatSeq(rs.getInt("BAT_SEQ"));
		ret.setPid(rs.getInt("PID"));
		ret.setBatName(rs.getString("BAT_NAME"));
		ret.setBatSts(rs.getInt("BAT_STATUS"));
		ret.setStartDtm(rs.getTimestamp("START_DATE_TIME"));
		ret.setStopDtm(rs.getTimestamp("STOP_DATE_TIME"));
		ret.setMsgTxt(rs.getString("MESSAGE"));
		return ret;
	}
}
