/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.web.servlet;


import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpSessionContext;

import org.apache.commons.chain2.web.MockEnumeration;



/**
 *  Mock Object for HttpSession (Version 2.3)
 */
public final class MockHttpSession implements HttpSession {

    private Map<String, Object> attributes = new HashMap<>();
    private ServletContext servletContext = null;

    /** constructor */
    public MockHttpSession() {
        super();
    }

    /**
     * @param ctx ServletContext
     */
    public MockHttpSession(final ServletContext ctx) {
        super();
        setServletContext(ctx);
    }


    // --------------------------------------------------------- Public Methods

    /**
     * @param ctx ServletContext
     */
    public void setServletContext(final ServletContext ctx) {
        this.servletContext = ctx;
    }


    // ---------------------------------------------------- HttpSession Methods

    /**
     * @see javax.servlet.http.HttpSession#getAttribute(java.lang.String)
     */
    @Override
    public Object getAttribute(final String name) {
        return this.attributes.get(name);
    }

    /**
     * @see javax.servlet.http.HttpSession#getAttributeNames()
     */
    @Override
    public Enumeration<String> getAttributeNames() {
        return new MockEnumeration<>(this.attributes.keySet().iterator());
    }

    /**
     * @see javax.servlet.http.HttpSession#getCreationTime()
     */
    @Override
    public long getCreationTime() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpSession#getId()
     */
    @Override
    public String getId() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpSession#getLastAccessedTime()
     */
    @Override
    public long getLastAccessedTime() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpSession#getMaxInactiveInterval()
     */
    @Override
    public int getMaxInactiveInterval() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpSession#getServletContext()
     */
    @Override
    public ServletContext getServletContext() {
        return this.servletContext;
    }

    /**
     * @deprecated deprecated
     */
    @Deprecated
    @Override
    public HttpSessionContext getSessionContext() {
        throw new UnsupportedOperationException();
    }

    /**
     * @deprecated deprecated
     */
    @Deprecated
    @Override
    public Object getValue(final String name) {
        throw new UnsupportedOperationException();
    }

    /**
     * @deprecated deprecated
     */
    @Deprecated
    @Override
    public String[] getValueNames() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpSession#invalidate()
     */
    @Override
    public void invalidate() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpSession#isNew()
     */
    @Override
    public boolean isNew() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpSession#putValue(java.lang.String, java.lang.Object)
     */
    @Override
    public void putValue(final String name, final Object value) {
        throw new UnsupportedOperationException();
    }


    /**
     * @see javax.servlet.http.HttpSession#removeAttribute(java.lang.String)
     */
    @Override
    public void removeAttribute(final String name) {
        this.attributes.remove(name);
    }

    /**
     * @see javax.servlet.http.HttpSession#removeValue(java.lang.String)
     */
    @Override
    public void removeValue(final String name) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpSession#setAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setAttribute(final String name, final Object value) {
        if (value == null) {
            this.attributes.remove(name);
        } else {
            this.attributes.put(name, value);
        }
    }

    /**
     * @see javax.servlet.http.HttpSession#setMaxInactiveInterval(int)
     */
    @Override
    public void setMaxInactiveInterval(final int interval) {
        throw new UnsupportedOperationException();
    }

}
