package common.sql;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;

/**
 * LineParsedNodeItemTest
 *
 * @author Tadashi Nakayama
 */
public class LineParsedNodeItemTest {

	/**
	 * テストターゲット取得
	 *
	 * @param arg 引数
	 * @return テストターゲット
	 */
	private LineParsedNodeItem getTarget(final String arg) {
		return new LineParsedNodeItem(arg);
	}

	/**
	 * 括弧パースOFFSET_LIMIT句
	 */
	@Test
	@DisplayName("括弧パースOFFSET_LIMIT句")
	public void testLineParsedNodeItem1() {
		final var item = getTarget(
				"OFFSET ((/*?Page*/1 - 1) * /*?Line*/20 ) LIMIT /*?Line*/20 \n");
		Assertions.assertEquals(
				"OFFSET ( ( /*?Page*/1 - 1 ) * /*?Line*/20 ) LIMIT /*?Line*/20", item.toString());
	}

	/**
	 * 括弧パースAND
	 */
	@Test
	@DisplayName("括弧パースAND")
	public void testLineParsedNodeItem2() {
		final var item = getTarget(
				"AND ((DT_DATTAI_YM = '      ' ) (( DT_DATTAI_YM >= ))/*:dtGyomu*/'200901' )");
		Assertions.assertEquals(
				"AND ( ( DT_DATTAI_YM = '      ' )( ( DT_DATTAI_YM >= ) ) /*:dtGyomu*/'200901' )",
				item.toString());
	}

	/**
	 * 括弧パース
	 */
	@Test
	@DisplayName("括弧パース")
	public void testLineParsedNodeItem3() {
		final var item = getTarget(
			"AND (DT_DATTAI_YM = '      ' OR DT_DATTAI_YM >= /*:dtGyomu*/'200901')");
		Assertions.assertEquals(
				"AND ( DT_DATTAI_YM = '      ' OR DT_DATTAI_YM >= /*:dtGyomu*/'200901' )",
				item.toString());
	}

	/**
	 * 括弧パースAND_OR
	 */
	@Test
	@DisplayName("括弧パースAND_OR")
	public void testLineParsedNodeItem4() {
		final var item = getTarget(
			"AND ((DT_DATTAI_YM = '      ' AND MN_PAY > /*:mnPay*/1 ) "
			+ "OR (DT_DATTAI_YM >= /*:dtGyomu*/'200901' ))");
		Assertions.assertEquals(
				"AND ( ( DT_DATTAI_YM = '      ' AND MN_PAY > /*:mnPay*/1 ) "
				+ "OR ( DT_DATTAI_YM >= /*:dtGyomu*/'200901' ) )",
				item.toString());
	}

	/**
	 * 日時必須パース
	 */
	@Test
	@DisplayName("日時必須パース")
	public void testLineParsedNodeItem5() {
		final var item = getTarget(
				"UPDATE_DATETIME = /*:DateTime*/TO_DATE('20091225','YYYYMMDD') ,");
		Assertions.assertEquals(
				"UPDATE_DATETIME = /*:DateTime*/TO_DATE( '20091225','YYYYMMDD' ) ,",
				item.toString());
	}

	/**
	 * 日時必須パースAND_OR句
	 */
	@Test
	@DisplayName("日時必須パースAND_OR句")
	public void testLineParsedNodeItem6() {
		final var item = getTarget(
			"AND ((DT_DATTAI_YM = /*:DateTime*/TO_DATE('20091225','YYYYMMDD') "
			+ "AND MN_PAY > /*:mnPay*/0 ) OR (DT_DATTAI_YM >= /*:dtGyomu*/'200901' ))");
		Assertions.assertEquals(
				"AND ( ( DT_DATTAI_YM = /*:DateTime*/TO_DATE( '20091225','YYYYMMDD' ) "
				+ "AND MN_PAY > /*:mnPay*/0 ) OR ( DT_DATTAI_YM >= /*:dtGyomu*/'200901' ) )",
				item.toString());
	}

	/**
	 * getBindString行末スペース
	 */
	@Test
	@DisplayName("getBindString行末スペース")
	public void testLineParsedNodeItem7() {
		final var item = getTarget("");
		Assertions.assertEquals("aaaaa#12345", item.getBindString("aaaaa#12345)", 0));
		Assertions.assertEquals("aaaaa#0.00012", item.getBindString("aaaaa#0.00012, ", 0));
		Assertions.assertEquals("aaaaa#-1234", item.getBindString("aaaaa#-1234 ", 0));
		Assertions.assertEquals("aaaaa#-1234.53", item.getBindString("aaaaa#-1234.53 ", 0));
		Assertions.assertEquals("aaaaa#-0.53", item.getBindString("aaaaa#-0.53 ", 0));
		Assertions.assertEquals("aaaaa#-0.00", item.getBindString("aaaaa#-0.00  ", 0));
	}
}
