/*
 *  The MIT License
 *
 *  Copyright 2011-2012 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger;

import android.app.ListActivity;
import android.content.Intent;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.os.Bundle;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.SimpleCursorAdapter;
import android.widget.TextView;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import org.routine_work.simple_battery_logger.db.BatteryHistoryDBHelper;
import org.routine_work.simple_battery_logger.db.DBConstants;
import org.routine_work.simple_battery_logger.service.BatteryLoggingService;
import org.routine_work.simple_battery_logger.service.ExportOneDayDataCsvFileService;
import org.routine_work.simple_battery_logger.util.Log;
import org.routine_work.simple_battery_logger.util.SystemResourceUtils;

public class DatabaseDateViewerActivity extends ListActivity
	implements SharedPreferences.OnSharedPreferenceChangeListener,
	SimpleCursorAdapter.ViewBinder, DBConstants
{

	/**
	 * Extra date as "yyyy-MM-dd"
	 */
	public static final String EXTRA_DATE = "date";
	private static final String LOG_TAG = "simple-battery-logger";
	private static final Map<Integer, Integer> STATUS_CODE_AND_RESOURCE_MAP = BatteryUtils.STATUS_CODE_AND_RESOURCE_MAP;
	private static final Map<Integer, Integer> PLUGGED_CODE_AND_RESOURCE_MAP = BatteryUtils.PLUGGED_CODE_AND_RESOURCE_MAP;
	private static final String[] MAPPING_FROM = new String[]
	{
		DBConstants.BatteryHistory.Columns.DATETIME,
		DBConstants.BatteryHistory.Columns.STATUS,
		DBConstants.BatteryHistory.Columns.PLUGGED,
		DBConstants.BatteryHistory.Columns.LEVEL,
		DBConstants.BatteryHistory.Columns.VOLTAGE,
		DBConstants.BatteryHistory.Columns.TEMPERATURE,
	};
	private static final int[] MAPPING_TO = new int[]
	{
		R.id.battery_history_datetime,
		R.id.battery_history_status,
		R.id.battery_history_plugged,
		R.id.battery_history_level,
		R.id.battery_history_voltage,
		R.id.battery_history_temperature,
	};
	private BatteryHistoryDBHelper dbHelper;
	private SQLiteDatabase db;
	private Cursor cursor;
	private SimpleCursorAdapter adapter;
	private SharedPreferences preferences;
	// parameter
	private Date targetDate;

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		MenuInflater menuInflater = getMenuInflater();

		menuInflater.inflate(R.menu.battery_history_menu, menu);
		menuInflater.inflate(R.menu.preference_menu, menu);
		menuInflater.inflate(R.menu.quit_menu, menu);

		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		boolean result = true;

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.preference_menuitem:
				Intent prefIntent = new Intent(this, BatteryLoggerPreferenceActivity.class);
				startActivity(prefIntent);
				break;
			case R.id.quit_menuitem:
				DashboardActivity.quitApplication(this);
				break;
			case R.id.export_csv_menuitem:
				exportCsvFile();
				break;
			default:
				if (itemId == SystemResourceUtils.getResourceId("android.R$id.home"))
				{
					Log.d(LOG_TAG, "home is clicked.");
					DashboardActivity.goDashboardActivity(this);
				}
				else
				{
					result = super.onOptionsItemSelected(item);
				}
				break;
		}

		return result;
	}

	/**
	 * Called when the activity is first created.
	 */
	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		Log.v(LOG_TAG, "Hello");
		super.onCreate(savedInstanceState);
		setContentView(R.layout.battery_history_list_activity);

		preferences = getSharedPreferences(getPackageName(), MODE_PRIVATE);

		initExtraParameter();
		initializeListData();

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onDestroy()
	{
		Log.v(LOG_TAG, "Hello");

		finalizeListData();
		super.onDestroy();

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onResume()
	{
		Log.v(LOG_TAG, "Hello");

		super.onResume();

		Log.d(LOG_TAG, "register OnSharedPreferenceChangeListener");
		preferences.registerOnSharedPreferenceChangeListener(this);

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onPause()
	{
		Log.v(LOG_TAG, "Hello");

		Log.d(LOG_TAG, "unregister OnSharedPreferenceChangeListener");
		preferences.unregisterOnSharedPreferenceChangeListener(this);

		super.onPause();

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public void onSharedPreferenceChanged(SharedPreferences prefs, String key)
	{
		Log.v(LOG_TAG, "onSharedPreferenceChanged(" + key + ") : Hello");

		if (BatteryLoggingService.PREFERENCE_KEY_ROW_ID.equals(key))
		{
			updateListData();
		}

		Log.v(LOG_TAG, "onSharedPreferenceChanged(" + key + ") : Bye");
	}

	@Override
	public boolean setViewValue(View view, Cursor cursor, int columnIndex)
	{

		String textValue = "?";
		if (columnIndex == cursor.getColumnIndex(BatteryHistory.Columns.STATUS))
		{
			int statusCode = cursor.getInt(columnIndex);
			if (STATUS_CODE_AND_RESOURCE_MAP.containsKey(statusCode))
			{
				int resourceId = STATUS_CODE_AND_RESOURCE_MAP.get(statusCode);
				textValue = getString(resourceId);
			}
		}
		else if (columnIndex == cursor.getColumnIndex(BatteryHistory.Columns.PLUGGED))
		{
			int pluggedCode = cursor.getInt(columnIndex);
			if (PLUGGED_CODE_AND_RESOURCE_MAP.containsKey(pluggedCode))
			{
				int resourceId = PLUGGED_CODE_AND_RESOURCE_MAP.get(pluggedCode);
				textValue = getString(resourceId);
			}
		}
		else if (columnIndex == cursor.getColumnIndex(BatteryHistory.Columns.TEMPERATURE))
		{
			int temperature = cursor.getInt(columnIndex);
			textValue = String.format("%.1f", (float) temperature / 10);
		}
		else
		{
			textValue = cursor.getString(columnIndex);
		}

		TextView textView = (TextView) view;
		textView.setText(textValue);

//		Log.v(LOG_TAG, "DatabaseDateViewerActivity#setViewValue() : Bye");
		return true;
	}

	private void initializeListData()
	{
		Log.v(LOG_TAG, "Hello");
		if (dbHelper == null)
		{
			dbHelper = new BatteryHistoryDBHelper(this);
		}

		db = dbHelper.getReadableDatabase();

		cursor = getBatteryHistoryCursor();

		adapter = new SimpleCursorAdapter(this,
			R.layout.battery_history_list_item, cursor, MAPPING_FROM, MAPPING_TO);
		adapter.setViewBinder(this);
		setListAdapter(adapter);
		Log.v(LOG_TAG, "Bye");
	}

	private void finalizeListData()
	{
		Log.v(LOG_TAG, "Hello");
		if (cursor != null)
		{
			cursor.close();
			cursor = null;
		}
		if (db != null)
		{
			db.close();
			db = null;
		}
		Log.v(LOG_TAG, "Bye");
	}

	private void updateListData()
	{
		Log.v(LOG_TAG, "Hello");
		if (db != null)
		{
			Cursor newCursor = getBatteryHistoryCursor();
			adapter.changeCursor(newCursor);
			cursor.close();
			cursor = newCursor;
		}
		Log.v(LOG_TAG, "Bye");
	}

	private void initExtraParameter()
	{
		String dateString = getIntent().getStringExtra(EXTRA_DATE);
		if (dateString != null)
		{
			SimpleDateFormat format = new SimpleDateFormat(DBConstants.DATE_FORMAT);
			try
			{
				targetDate = format.parse(dateString);
				this.setTitle(dateString);
			}
			catch (ParseException ex)
			{
				Log.w(LOG_TAG, "Failed in Parsing EXTRA_DATE", ex);
			}
		}
	}

	private Cursor getBatteryHistoryCursor()
	{
		Cursor result;

		if (targetDate != null)
		{
			result = dbHelper.getOneDayBatteryHistoryCursor(db, targetDate, DBConstants.SORT_DESCENDING);
		}
		else
		{
			result = dbHelper.getAllBatteryHistoryCursor(db);
		}

		return result;
	}

	private void exportCsvFile()
	{
		Log.v(LOG_TAG, "Hello");

		if (targetDate != null)
		{
			Log.d(LOG_TAG, "exportCsvFile() : targetDate => " + targetDate);
			SimpleDateFormat format = new SimpleDateFormat(DBConstants.DATE_FORMAT);
			String targetDateString = format.format(targetDate);
			Intent intent = new Intent(this, ExportOneDayDataCsvFileService.class);
			intent.putExtra(ExportOneDayDataCsvFileService.EXTRA_DATE, targetDateString);
			startService(intent);
		}

		Log.v(LOG_TAG, "Bye");
	}
}
