/*
 *  The MIT License
 *
 *  Copyright 2011-2012 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger;

import android.app.ListActivity;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.os.Bundle;
import android.view.ContextMenu;
import android.view.ContextMenu.ContextMenuInfo;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.AdapterView;
import android.widget.AdapterView.AdapterContextMenuInfo;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.ListView;
import android.widget.SimpleCursorAdapter;
import android.widget.TextView;
import org.routine_work.simple_battery_logger.db.BatteryHistoryDBHelper;
import org.routine_work.simple_battery_logger.db.DBConstants;
import org.routine_work.simple_battery_logger.service.BatteryLoggingService;
import org.routine_work.simple_battery_logger.service.ExportOneDayDataCsvFileCompletedReceiver;
import org.routine_work.simple_battery_logger.service.ExportOneDayDataCsvFileService;
import org.routine_work.simple_battery_logger.util.Log;
import org.routine_work.simple_battery_logger.util.SystemResourceUtils;

public class DatabaseDateListActivity extends ListActivity
	implements SharedPreferences.OnSharedPreferenceChangeListener,
	OnItemClickListener, DBConstants
{

	private static final int REQUEST_CODE_BATTERY_HISTORY_LIST_ACTIVITY = 100;
	private static final String LOG_TAG = "simple-battery-logger";
	private static final String[] MAPPING_FROM = new String[]
	{
		BatteryHistoryDate.Columns.DATE,
	};
	private static final int[] MAPPING_TO = new int[]
	{
		android.R.id.text1
	};
	private BatteryHistoryDBHelper dbHelper;
	private SQLiteDatabase db;
	private Cursor cursor;
	private SimpleCursorAdapter adapter;
	private SharedPreferences preferences;
	private ExportOneDayDataCsvFileCompletedReceiver exportOneDayDataCsvFileCompletedReceiver;

	/**
	 *
	 * @param savedInstanceState
	 */
	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		setContentView(R.layout.simple_list_activity);

		ListView listView = getListView();
		listView.setOnItemClickListener(this);
		registerForContextMenu(listView);

		initializeListData();

		preferences = getSharedPreferences(getPackageName(), MODE_PRIVATE);

		Log.d(LOG_TAG, "register broadcast register for export CSV completed.");
		IntentFilter intentFilter;
		intentFilter = new IntentFilter();
		intentFilter.addAction(ExportOneDayDataCsvFileCompletedReceiver.EXPORT_ONE_DAY_DATA_CSV_FILE_COMPLETED_ACTION);
		exportOneDayDataCsvFileCompletedReceiver = new ExportOneDayDataCsvFileCompletedReceiver();
		registerReceiver(exportOneDayDataCsvFileCompletedReceiver, intentFilter);
	}

	@Override
	protected void onDestroy()
	{
		Log.v(LOG_TAG, "Hello");

		finalizeListData();

		Log.d(LOG_TAG, "unregister broadcast register for export CSV completed.");
		unregisterReceiver(exportOneDayDataCsvFileCompletedReceiver);

		super.onDestroy();
		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onResume()
	{
		Log.v(LOG_TAG, "Hello");
		super.onResume();

		Log.d(LOG_TAG, "register OnSharedPreferenceChangeListener");
		preferences.registerOnSharedPreferenceChangeListener(this);

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onPause()
	{
		Log.v(LOG_TAG, "Hello");

		Log.d(LOG_TAG, "unregister OnSharedPreferenceChangeListener");
		preferences.unregisterOnSharedPreferenceChangeListener(this);

		super.onPause();
		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public void onSharedPreferenceChanged(SharedPreferences prefs, String key)
	{
		Log.v(LOG_TAG, "Hello");
		Log.d(LOG_TAG, "key " + key);

		if (BatteryLoggingService.PREFERENCE_KEY_ROW_ID.equals(key))
		{
			updateListData();
		}

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public void onItemClick(AdapterView<?> parentView, View itemView, int position, long id)
	{
		Log.v(LOG_TAG, "Hello");
		if (itemView instanceof TextView)
		{
			TextView dateTextView = (TextView) itemView;
			String dateTextString = dateTextView.getText().toString();
			Log.d(LOG_TAG, "clicked date text => " + dateTextString);
			if (dateTextString != null && dateTextString.length() > 0)
			{
				Intent intent = new Intent(this, DatabaseDateViewerActivity.class);
				intent.putExtra(DatabaseDateViewerActivity.EXTRA_DATE, dateTextString);
				startActivity(intent);
			}
		}
		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		Log.v(LOG_TAG, "Hello");

		MenuInflater menuInflater = getMenuInflater();
		menuInflater.inflate(R.menu.preference_menu, menu);
		menuInflater.inflate(R.menu.quit_menu, menu);

		Log.v(LOG_TAG, "Bye");
		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		boolean result = true;
		Log.v(LOG_TAG, "Hello");

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.preference_menuitem:
				Intent prefIntent = new Intent(this, BatteryLoggerPreferenceActivity.class);
				startActivity(prefIntent);
				break;
			case R.id.quit_menuitem:
				Log.d(LOG_TAG, "quit_menuitem is selected.");
				DashboardActivity.quitApplication(this);
				break;
			default:
				if (itemId == SystemResourceUtils.getResourceId("android.R$id.home"))
				{
					Log.d(LOG_TAG, "home is clicked.");
					DashboardActivity.goDashboardActivity(this);
				}
				else
				{
					result = super.onOptionsItemSelected(item);
				}
				break;
		}

		Log.v(LOG_TAG, "Bye");
		return result;
	}

	@Override
	public void onCreateContextMenu(ContextMenu menu, View view, ContextMenuInfo menuInfo)
	{
		Log.v(LOG_TAG, "Hello");
		Log.d(LOG_TAG, "view => " + view);
		super.onCreateContextMenu(menu, view, menuInfo);

		MenuInflater menuInflater = getMenuInflater();
		menuInflater.inflate(R.menu.battery_history_menu, menu);
		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public boolean onContextItemSelected(MenuItem item)
	{
		boolean result;
		Log.v(LOG_TAG, "Hello");

		AdapterContextMenuInfo menuInfo = (AdapterContextMenuInfo) item.getMenuInfo();
		Log.d(LOG_TAG, "menuInfo.position => " + menuInfo.position);
		Log.d(LOG_TAG, "menuInfo.id => " + menuInfo.id);
		Log.d(LOG_TAG, "menuInfo.targetView => " + menuInfo.targetView);

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.export_csv_menuitem:
				if (menuInfo.targetView instanceof TextView)
				{
					TextView textView = (TextView) menuInfo.targetView;
					String dateText = textView.getText().toString();
					exportCsvFile(dateText);
				}
				result = true;
				break;
			default:
				result = super.onContextItemSelected(item);
		}

		Log.v(LOG_TAG, "Bye");
		return result;
	}

	private void initializeListData()
	{
		Log.v(LOG_TAG, "Hello");
		if (dbHelper == null)
		{
			dbHelper = new BatteryHistoryDBHelper(this);
		}

		db = dbHelper.getReadableDatabase();

		cursor = dbHelper.getBatteryHistoryDateListCursor(db);

		adapter = new SimpleCursorAdapter(this,
			android.R.layout.simple_list_item_1,
			cursor, MAPPING_FROM, MAPPING_TO);
		setListAdapter(adapter);
		Log.v(LOG_TAG, "Bye");
	}

	private void finalizeListData()
	{
		Log.v(LOG_TAG, "Hello");
		if (cursor != null)
		{
			cursor.close();
			cursor = null;
		}
		if (db != null)
		{
			db.close();
			db = null;
		}
		Log.v(LOG_TAG, "Bye");
	}

	private void updateListData()
	{
		Log.v(LOG_TAG, "Hello");
		if (db != null)
		{
			Cursor newCursor = dbHelper.getBatteryHistoryDateListCursor(db);
			adapter.changeCursor(newCursor);
			cursor.close();
			cursor = newCursor;
		}
		Log.v(LOG_TAG, " Bye");
	}

	private void exportCsvFile(String dateText)
	{
		Log.v(LOG_TAG, "Hello");

		Log.d(LOG_TAG, "dateText => " + dateText);
		if (dateText != null)
		{
			Intent intent = new Intent(this, ExportOneDayDataCsvFileService.class);
			intent.putExtra(ExportOneDayDataCsvFileService.EXTRA_DATE, dateText);
			startService(intent);
		}

		Log.v(LOG_TAG, "Bye");
	}
}
