// Programming 2D Games
// Copyright (c) 2011 by: 
// Charles Kelly
// spacewar.cpp Scrolling Bitmap Demo
// This class is the core of the game

#include "spaceWar.h"

//=============================================================================
// Constructor
//=============================================================================
Spacewar::Spacewar()
{
    menuOn = true;
    countDownOn = false;
    roundOver = false;
}

//=============================================================================
// Destructor
//=============================================================================
Spacewar::~Spacewar()
{
    releaseAll();           // call onLostDevice() for every graphics item
}

//=============================================================================
// Initializes the game
// Throws GameError on error
//=============================================================================
void Spacewar::initialize(HWND hwnd)
{
    Game::initialize(hwnd); // throws GameError

    // initialize DirectX fonts
    fontBig.initialize(graphics, spacewarNS::FONT_BIG_SIZE, false, false, spacewarNS::FONT);
    fontBig.setFontColor(spacewarNS::FONT_COLOR);

    // menu texture
    if (!menuTexture.initialize(graphics,MENU_IMAGE))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing menu texture"));

    // space texture
    if (!spaceTexture.initialize(graphics,SPACE_IMAGE))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing space texture"));

    // stars texture
    if (!starsTexture.initialize(graphics,STARS_IMAGE))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing space texture"));

    // game textures
    if (!textures1.initialize(graphics,TEXTURES1_IMAGE))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing game textures"));

    // menu image
    if (!menu.initialize(graphics,0,0,0,&menuTexture))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing menu"));

    // space image
    if (!space[0].initialize(graphics,0,0,0,&spaceTexture))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing space"));
    space[0].setScale((float)SPACE_SCALE);
    if (!space[1].initialize(graphics,0,0,0,&starsTexture))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing space"));
    space[1].setScale((float)SPACE_SCALE);

    // ship1
    if (!ship1.initialize(this,shipNS::WIDTH,shipNS::HEIGHT,shipNS::TEXTURE_COLS,&textures1))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing ship1"));

    ship1.setFrames(shipNS::SHIP1_START_FRAME, shipNS::SHIP1_END_FRAME);
    ship1.setCurrentFrame(shipNS::SHIP1_START_FRAME);
    ship1.setColorFilter(SETCOLOR_ARGB(255,230,230,255));   // light blue, used for shield and torpedo
    ship1.setMass(shipNS::MASS);
    // Start ship in center
    ship1.setX(GAME_WIDTH/2 - shipNS::WIDTH);
    ship1.setY(GAME_HEIGHT/2 - shipNS::HEIGHT);
}

//=============================================================================
// Start a new round of play
//=============================================================================
void Spacewar::roundStart()
{
    space[0].setX(0);
    space[0].setY(0);
    space[1].setX(SPACE_WIDTH/2);
    space[1].setY(SPACE_HEIGHT/2);

    // Start ship in center
    ship1.setX(GAME_WIDTH/2 - shipNS::WIDTH);
    ship1.setY(GAME_HEIGHT/2 - shipNS::HEIGHT);

    ship1.setDegrees(0);
    countDownTimer = spacewarNS::COUNT_DOWN;
    countDownOn = true;
    roundOver = false;
}

//=============================================================================
// Update all game items
//=============================================================================
void Spacewar::update()
{
    if (menuOn)
    {
        if (input->anyKeyPressed())
        {
            menuOn = false;
            input->clearAll();
            roundStart();
        }
    } 
    else if(countDownOn)
    {
        countDownTimer -= frameTime;
        if(countDownTimer <= 0)
            countDownOn = false;
    } 
    else 
    {
        if (ship1.getActive())
        {
            if (input->isKeyDown(SHIP1_FORWARD_KEY) || input->getGamepadDPadUp(0))   // if engine on
            {
                ship1.setEngineOn(true);
                audio->playCue(ENGINE1);
            }
            else
            {
                ship1.setEngineOn(false);
                audio->stopCue(ENGINE1);
            }
            ship1.rotate(shipNS::NONE);
            if (input->isKeyDown(SHIP1_LEFT_KEY) || input->getGamepadDPadLeft(0))   // if turn ship1 left
                ship1.rotate(shipNS::LEFT);
            if (input->isKeyDown(SHIP1_RIGHT_KEY) || input->getGamepadDPadRight(0)) // if turn ship1 right
                ship1.rotate(shipNS::RIGHT);
        }
        if(roundOver)
        {
            roundTimer -= frameTime;
            if(roundTimer <= 0)
                roundStart();
        }
    }

    // Update the entities
    ship1.update(frameTime);
    // move space in X direction opposite ship
    space[0].setX(space[0].getX() - frameTime * ship1.getVelocity().x);
    space[1].setX(space[1].getX() - frameTime * ship1.getVelocity().x * 1.2f);
    // move space in Y direction opposite ship
    space[0].setY(space[0].getY() - frameTime * ship1.getVelocity().y);
    space[1].setY(space[1].getY() - frameTime * ship1.getVelocity().y * 1.2f);

    // Wrap space image around at edge
    // if left edge of space > screen left edge
    for (int h = 0;h < SPACE_MAX;++h) {
        if (space[h].getX() > 0)               
            // move space image left by SPACE_WIDTH
            space[h].setX(space[h].getX() - SPACE_WIDTH);
        // if space image off screen left
        if (space[h].getX() < -SPACE_WIDTH)
            // move space image right by SPACE_WIDTH
            space[h].setX(space[h].getX() + SPACE_WIDTH);
        // if top edge of space > screen top edge
        if (space[h].getY() > 0)
            // move space image up by SPACE_HEIGHT
            space[h].setY(space[h].getY() - SPACE_HEIGHT);
        // if space image off screen top
        if (space[h].getY() < -SPACE_HEIGHT)
            // move space image down by SPACE_IMAGE
            space[h].setY(space[h].getY() + SPACE_HEIGHT);
    }
}

//=============================================================================
// Artificial Intelligence
//=============================================================================
void Spacewar::ai()
{}

//=============================================================================
// Handle collisions
//=============================================================================
void Spacewar::collisions()
{
}

//=============================================================================
// Render game items
//=============================================================================
void Spacewar::render()
{
    graphics->spriteBegin();                // begin drawing sprites

    // Wrap space image around at edges
    for (int h = 0;h < SPACE_MAX;++h) {
        float x = space[h].getX();
        float y = space[h].getY();
        space[h].draw();                           // draw at current location
        // if space image right edge visible
        if (space[h].getX() < -SPACE_WIDTH + (int)GAME_WIDTH)
        {
            space[h].setX(space[h].getX() + SPACE_WIDTH); // wrap around to left edge
            space[h].draw();                           // draw again
        }
        // if space image bottom edge visible
        if (space[h].getY() < -SPACE_HEIGHT + (int)GAME_HEIGHT)
        {
            space[h].setY(space[h].getY() + SPACE_HEIGHT); // wrap around to top edge
            space[h].draw();                           // draw again
            space[h].setX(x);                          // restore x position
            // if space image right edge visible
            // wrap around to left edge
            if (x < -SPACE_WIDTH + (int)GAME_WIDTH)
                space[h].draw();                       // draw again
        }
        space[h].setX(x);      // restore x position
        space[h].setY(y);      // restore y position
    }

    ship1.draw();       // draw the spaceship

    if(menuOn)
        menu.draw();
    if(countDownOn)
    {
        _snprintf_s(buffer, spacewarNS::BUF_SIZE, "%d", (int)(ceil(countDownTimer)));
        fontBig.print(buffer,spacewarNS::COUNT_DOWN_X,spacewarNS::COUNT_DOWN_Y);
    }

    graphics->spriteEnd();                  // end drawing sprites
}

//=============================================================================
// The graphics device was lost.
// Release all reserved video memory so graphics device may be reset.
//=============================================================================
void Spacewar::releaseAll()
{
    menuTexture.onLostDevice();
    spaceTexture.onLostDevice();
    starsTexture.onLostDevice();
    textures1.onLostDevice();
    fontScore.onLostDevice();
    fontBig.onLostDevice();

    Game::releaseAll();
    return;
}

//=============================================================================
// The grahics device has been reset.
// Recreate all surfaces.
//=============================================================================
void Spacewar::resetAll()
{
    fontBig.onResetDevice();
    fontScore.onResetDevice();
    textures1.onResetDevice();
    spaceTexture.onResetDevice();
    starsTexture.onResetDevice();
    menuTexture.onResetDevice();

    Game::resetAll();
    return;
}
