/*
 * Copyright (c) 2009-2010 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;
import java.text.Format;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;
import jp.sf.orangesignal.csv.filters.CsvNamedValueFilter;
import jp.sf.orangesignal.csv.manager.CsvBeanManager;

/**
 * Java プログラム要素のリストと区切り文字形式データアクセスを行うハンドラを提供します。
 * 
 * @author 杉澤 浩二
 * @see CsvBeanManager
 */
public class BeanListHandler<T> extends BeanListHandlerSupport<T, BeanListHandler<T>> {

	/**
	 * Java プログラム要素へデータを設定する名前群を保持します。
	 */
	private String[] includeNames;

	/**
	 * Java プログラム要素へデータを設定しない名前群を保持します。
	 */
	private String[] excludeNames;

	/**
	 * 区切り文字形式データフィルタを保持します。
	 */
	private CsvNamedValueFilter valueFilter;

	// ------------------------------------------------------------------------

	/**
	 * コンストラクタです。
	 * 
	 * @param type Java プログラム要素の型
	 * @throws IllegalArgumentException <code>type</code> が <code>null</code> の場合
	 */
	public BeanListHandler(final Class<T> type) {
		super(type);
	}

	// ------------------------------------------------------------------------

	/**
	 * Java プログラム要素へデータを設定する名前群を設定します。
	 * 
	 * @param names Java プログラム要素へデータを設定する名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException Java プログラム要素へデータを設定しない名前群が存在する場合
	 */
	public BeanListHandler<T> includes(final String... names) {
		if (excludeNames != null && excludeNames.length > 0) {
			throw new IllegalArgumentException("Only includes or excludes may be specified.");
		}
		this.includeNames = names;
		return this;
	}

	/**
	 * Java プログラム要素へデータを設定しない名前群を設定します。
	 * 
	 * @param names Java プログラム要素へデータを設定しない名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException Java プログラム要素へデータを設定する名前群が存在する場合
	 */
	public BeanListHandler<T> excludes(final String... names) {
		if (includeNames != null && includeNames.length > 0) {
			throw new IllegalArgumentException("Only includes or excludes may be specified.");
		}
		this.excludeNames = names;
		return this;
	}

	/**
	 * 指定された名前が Java プログラム要素としてデータを設定すべき名前かどうかを返します。
	 * 
	 * @param name 名前
	 * @return Java プログラム要素としてデータを設定すべき名前の場合は <code>true</code> それ以外の場合は <code>false</code>
	 */
	private boolean isTargetName(final String name) {
		if (excludeNames != null && excludeNames.length > 0) {
			for (final String propertyName : excludeNames) {
				if (propertyName.equals(name))
					return false;
			}
			return true;
		}
		if (includeNames != null && includeNames.length > 0) {
			for (final String propertyName : includeNames) {
				if (propertyName.equals(name))
					return true;
			}
			return false;
		}
		return true;
	}

	/**
	 * 指定された Java プログラム要素のフィールドを処理するフォーマットオブジェクトを設定します。
	 * 
	 * @param name Java プログラム要素のフィールド名
	 * @param format フィールドを処理するフォーマットオブジェクト
	 * @return このオブジェクトへの参照
	 * @since 1.2
	 */
	public BeanListHandler<T> format(final String name, final Format format) {
		setValueParser(name, format);
		setValueFormatter(name, format);
		return this;
	}

	/**
	 * 区切り文字形式データフィルタを設定します。
	 * 
	 * @param filter 区切り文字形式データフィルタ
	 * @return このオブジェクトへの参照
	 * @since 1.2.3
	 */
	public BeanListHandler<T> filter(final CsvNamedValueFilter filter) {
		this.valueFilter = filter;
		return this;
	}

	// ------------------------------------------------------------------------

	@Override
	public List<T> load(final CsvReader reader, final boolean ignoreScalar) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		// ヘッダ部を処理します。
		final List<String> columnNames = reader.readValues();
		if (columnNames == null) {
			// ヘッダがない場合は例外をスローします。
			throw new IOException("No header is available");
		}

		// データ部を処理します。
		final List<T> results = new ArrayList<T>();
		final int columnCount = columnNames.size();
		final boolean order = ignoreScalar || (orders != null && !orders.isEmpty());
		int offset = 0;
		List<String> values;
		while ((values = reader.readValues()) != null && (order || limit <= 0 || results.size() < limit)) {
			if (valueFilter != null && !valueFilter.accept(columnNames, values)) {
				continue;
			}
			if (beanFilter == null && !order && offset < this.offset) {
				offset++;
				continue;
			}
			final T bean = createBean();
			final int len = Math.min(columnCount, values.size());
			for (int pos = 0; pos < len; pos++) {
				final String name = columnNames.get(pos);
				if (!isTargetName(name)) {
					continue;
				}
				final Field f = getField(getType(), name);
				final Object o = stringToObject(f, values.get(pos));
				if (o != null) {
					setFieldValue(bean, f, o);
				}
			}
			if (beanFilter != null) {
				if (!beanFilter.accept(bean)) {
					continue;
				}
				if (!order && offset < this.offset) {
					offset++;
					continue;
				}
			}
			results.add(bean);
		}

		if (ignoreScalar || !order) {
			return results;
		}
		return processScalar(results);
	}

	@Override
	public void save(final List<T> list, final CsvWriter writer) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		// ヘッダ部を処理します。
		final List<String> columnNames = new ArrayList<String>();
		for (final Field f : getType().getDeclaredFields()) {
			final String name = f.getName();
			if (isTargetName(name)) {
				columnNames.add(name);
			}
		}
		writer.writeValues(columnNames);

		// データ部を処理します。
		final int columnCount = columnNames.size();
		for (final T bean : list) {
			// 要素が null の場合は null 出力します。
			if (bean == null) {
				writer.writeValues(null);
				continue;
			} else if (beanFilter != null && !beanFilter.accept(bean)) {
				continue;
			}

			final String[] values = new String[columnCount];
			for (int i = 0; i < columnCount; i++) {
				final String name = columnNames.get(i);
				if (name == null) {
					continue;
				}
				final Field f = getField(bean.getClass(), name);
				values[i] = objectToString(name, getFieldValue(bean, f));
			}
			final List<String> _values = Arrays.asList(values);
			if (valueFilter != null && !valueFilter.accept(columnNames, _values)) {
				continue;
			}
			writer.writeValues(_values);
		}
	}

}
