#pragma once
/*
  ==============================================================================

   This file is part of the S.F.Tracker
   Copyright 2005-7 by Satoshi Fujiwara.

   S.F.Tracker can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   S.F.Tracker is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with S.F.Tracker; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ==============================================================================
*/
/*                                                                            
                                                                              
                           **** sequencer_base FORMAT ****                                 
                                                                              
                                                                              
            TIME +------------------------------------------>                 
           +--+------------+------------------------------------+             
      TRK  |  |  PATTERN   |             SEQUENCE               |             
       +   +--+            ++-----------------------------+-----+             
       |   |  |            ||PATTERN                      |     |             
       |   +--+      +------+-----+---------+-------------+-----+             
       |   |  |      |PATTERN     |                             |             
       |   +--+------+-----+------+--------+--------------------+             
       |   |               |PATTERN        |                    |             
       |   +---------------+Max:256 tracks +--------------------+             
       |   |               |               |                    |             
       |   +---------------+               +--------------------+             
       |   |               |               |                    |             
       |   +---------------+               +--------------------+             
       |   |               |               |                    |             
       |   +---------------+---------------+--------------------+             
       |   |                                                    |             
       |   +----------------------------------------------------+             
       V   |                                                    |             
           +----------------------------------------------------+             
                                                                              
      max 256 tracks.                                                           

    ** TICK **
    
    tick(64bit int) = (bpm * (machine sample rate)) / ((interface sample rate) * 120.0) * 4294967296.0
    
    ** VOLUME **

    ouput volume(float) = velocity(0.0f-1.0f) 
        * track_volume(0.0f-1.0f) 
        * pattern_volume(0.0f-1.0f) 
        * sequence_volume(0.0f-1.0f)
    
    ** PITCH **

    scale(boost::uint32_t, 0x00000000 - 0x007f0000)
    play pitch(Hz) = 440(Hz) * 2^((scale - 0x40 + track_transpose + pattern_transpose + sequence_transpose ) / 12 ) + pitch_shift; 
        
    */
#include "machine.h"

namespace sf {
    namespace model {
        namespace machine {
            struct machines;
            struct pattern_command;

            typedef boost::ptr_multimap<boost::uint32_t,pattern_command> seq_buf_type;


            struct pattern_command {
                
                friend boost::serialization::access;

                pattern_command(const boost::uint32_t step_time_value,machine_interface* mac,const boost::uint32_t midi_ch = 0) 
                    : m_step_time(step_time_value),m_machine(mac),m_midi_channel(midi_ch) {}
                pattern_command(const pattern_command& src ): m_step_time(src.m_step_time),m_machine(src.m_machine),m_midi_channel(src.m_midi_channel) {}

                const boost::uint32_t step_time() const { return m_step_time;}
                void step_time(const boost::uint32_t value) {m_step_time = value;}

                const boost::uint32_t midi_channel() const { return m_midi_channel;}
                void midi_channel(const boost::uint32_t midi_ch) {m_midi_channel = midi_ch;}

                machine_interface* machine_ref() const { return m_machine;}
                void machine_ref(machine_interface * mac) {m_machine = mac;}

                void sequence(const boost::uint32_t delta_tick){ sequence_(delta_tick);}

                void insert_sequence_buf(boost::uint32_t time,seq_buf_type& seq_buf,machine_interface* sequencer_ = NULL) const
                {   
                    insert_sequence_buf_(time,seq_buf,sequencer_); 
                }

                pattern_command* clone() const  { return clone_(); }
                const std::wstring& name() {return name_();}
                const std::wstring& description() {return description_();}

            protected:
                virtual void sequence_(const boost::uint32_t delta_tick){ m_machine->sequence(delta_tick,*this);}
                virtual void insert_sequence_buf_ (boost::uint32_t time,seq_buf_type& seq_buf,machine_interface* sequencer_) const = 0;
                virtual pattern_command* clone_() const  = 0;
                virtual const std::wstring&  name_() const= 0;
                virtual const std::wstring&  description_() const = 0;

                boost::uint32_t m_step_time;
                boost::uint32_t m_midi_channel;// midi channel
//                boost::uint32_t m_machine;
                machine_interface* m_machine;
            private:
                template <class Archive>
                void serialize(Archive& ar, const unsigned int version)
                {
//                    ar & make_nvp("step_time",m_step_time);
//                    ar & make_nvp("midi_ch",m_midi_channel);
                }
            };

            enum note_no {
                C0,CS0,D0,DS0,E0,F0,FS0,G0,GS0,A0,AS0,B0,
                C1,CS1,D1,DS1,E1,F1,FS1,G1,GS1,A1,AS1,B1,
                C2,CS2,D2,DS2,E2,F2,FS2,G2,GS2,A2,AS2,B2,
                C3,CS3,D3,DS3,E3,F3,FS3,G3,GS3,A3,AS3,B3,
                C4,CS4,D4,DS4,E4,F4,FS4,G4,GS4,A4,AS4,B4,
                C5,CS5,D5,DS5,E5,F5,FS5,G5,GS5,A5,AS5,B5,
                C6,CS6,D6,DS6,E6,F6,FS6,G6,GS6,A6,AS6,B6,
                C7,CS7,D7,DS7,E7,F7,FS7,G7,GS7,A7,AS7,B7,
                C8,CS8,D8,DS8,E8,F8,FS8,G8,GS8,A8,AS8,B8,
                C9,CS9,D9,DS9,E9,F9,FS9,G9,GS9,A9,AS9,B9
            };

            struct track_info 
            {
                friend boost::serialization::access;

                const boost::uint32_t transpose() const throw() {return m_transpose;}
                void transpose(const boost::uint32_t value) throw() {m_transpose = value;}

                const bool is_mute() const throw() {return m_mute;}
                void mute(const bool value) throw() {m_mute = value;}

            private:
                boost::uint32_t m_transpose;
                bool m_mute;
                template <class Archive>
                void serialize(Archive& ar, const unsigned int version)
                {
                    ar & BOOST_SERIALIZATION_NVP(m_transpose);
                    ar & BOOST_SERIALIZATION_NVP(m_mute);

                }

            };

            typedef boost::ptr_vector< track_info > tracks_type;

            struct song 
            {
                song(mutex_type& mutex,machines & machines_ref);

                struct pattern 
                {
                    friend boost::serialization::access;
            
                    typedef boost::ptr_multimap<boost::uint32_t,pattern_command> rows_type;

                    typedef boost::ptr_vector<rows_type> tracks_type;

                    pattern() : m_start_track(0),m_transpose(0) {}

                    const tracks_type& tracks() const {return m_tracks;}
                   
                    const float transpose(){return m_transpose;}
                    void transpose(const float value){m_transpose = value;}

                    const int start_track() const throw() {return m_start_track;}
                    void start_track(const int value ){m_start_track = value;}

                private:
                    tracks_type m_tracks;
                    int m_start_track;
                    float m_transpose;
                    template <class Archive>
                    void serialize(Archive& ar, const unsigned int version)
                    {
                        ar & BOOST_SERIALIZATION_NVP(m_tracks);
                        ar & BOOST_SERIALIZATION_NVP(m_start_track);
                        ar & BOOST_SERIALIZATION_NVP(m_transpose);
                    }
                };
                
                typedef boost::ptr_vector<pattern> patterns_type;
                typedef std::multimap<boost::uint32_t,boost::uint32_t> sequences_type;
                
                //song();
                virtual ~song();
     
                /** g̕쐬 */
                song* clone() const { return NULL;};

                void bpm(const boost::uint32_t bpm) throw(){m_bpm = bpm;}
                const boost::uint32_t bpm() const throw(){return m_bpm;}

                const patterns_type& patterns() const throw() {return m_patterns;}
                const sequences_type& sequences() const throw() {return m_sequences;}

                void track_size(const boost::uint32_t value) throw() {m_track_size = value;}
                const boost::uint32_t track_size() const throw(){return m_track_size;}

                const tracks_type& initial_tracks() const throw() {return m_initial_tracks;}


            private:

                tracks_type m_initial_tracks;

                patterns_type m_patterns;
                sequences_type m_sequences;
                boost::uint32_t m_bpm;
                boost::uint32_t m_track_size;
                machines& m_machines;
                mutex_type& m_mutex;
            };

            template <class CpuType = cpu_default >
            struct sequencer_base : abstract_machine
            {
                static const boost::uint32_t start_tick_default =  0x00000000;
                static const boost::uint32_t end_tick_default =    0xffffffff;
                typedef sequencer_base<CpuType> this_type; 
              
                sequencer_base(mutex_type& mutex,song& sng);

                virtual ~sequencer_base();
     
                /** g̕쐬 */
                this_type * clone() const { return NULL;};

                void bpm(const boost::uint32_t bpm) throw(){m_bpm = bpm;}
                const boost::uint32_t bpm() const throw(){return m_bpm;}

                void seek(const boost::uint32_t pos);
                const boost::uint32_t current_pos(){return m_seq_buf_iter->second->step_time();}
 
                const tracks_type& tracks() const throw() {return m_tracks;}

                const song* song_ref() const {return &m_song;}
                 
                // do sequence
                void sequence(const boost::uint32_t start_tick,const boost::uint32_t end_tick);
                boost::uint32_t build_sequence_buf(std::vector<machine_interface*>& macs,const bool clear_buffer = true,const boost::uint32_t stat_tick = start_tick_default,const boost::uint32_t end_tick = end_tick_default);
                // do sequence
                boost::uint32_t build_sequence_buf(seq_buf_type& seq_buf,std::vector<machine_interface*>& macs ,const bool clear_buffer = true,const boost::uint32_t stat_tick = start_tick_default,const boost::uint32_t end_tick = end_tick_default);

            private:
                virtual void init_() 
                {
                    m_seq_buf_iter = m_seq_buf.begin();
                };

                virtual void reset_()
                {
                    //build_sequence_buf();
                };

                virtual float * process_(cable& cable_to,const int out_buffer_count,float * buffer = NULL,const bool replace = false){return buffer;}
                virtual const std::wstring& name_() const throw() {return m_name;}
                virtual const std::wstring& description_() const throw(){return m_description;}
                virtual const std::wstring& vendor_() const throw() {return m_vendor;}
                virtual void configure_(){}

                boost::uint32_t m_bpm;
                song& m_song;

                seq_buf_type::iterator m_seq_buf_iter;
                seq_buf_type m_seq_buf;
                tracks_type m_tracks;

                static std::wstring m_name;
                static std::wstring m_vendor;
                // static std::wstring m_author;
                static std::wstring m_description;
              
            };

            typedef sequencer_base<> sequencer;

            inline void sequence(seq_buf_type& seq_buf,seq_buf_type::iterator& it,const boost::uint32_t start_tick,const boost::uint32_t end_tick)
            {
                // obt@ĂR}h̃V[PX
                {
                    //seq_buf_type::iterator itr_s_ = m_seq_buf.upper_bound(start_tick);
                    if(it != seq_buf.end() && it->first >= start_tick)
                    {
                        //pattern::rows_type::iterator itr_e_ = seq_buf.lower_bound(end_tick);
                        //
                        //if(itr_e_ != seq_buf.end())
                        //{
                        //    ++itr_e_;
                        //}

                        while(it != seq_buf.end() && it->second->step_time() < end_tick)
                        {
                            //  Tick̎Zo
                            boost::uint32_t delta_tick_ = it->second->step_time() - start_tick;
                            it->second->sequence(delta_tick_);
                            //mac_.sequence(delta_tick_,*it);
                            ++it;
                        }

                        //if(itr_e_ != seq_buf.end())
                        //{
                        //    --itr_e_;
                        //}

                        //seq_buf.erase(itr_s_,itr_e_);

                    }


                }

            };

        }// machine
    }// model
}// sf

namespace boost {
    namespace serialization {
    }
}