/*!
******************************************************************************

	@file	syscall.h

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#ifndef __SYSCALL_H__
#define __SYSCALL_H__

#ifndef u8_u16_u32_u64
#define u8_u16_u32_u64
typedef unsigned char		u8;
typedef unsigned short		u16;
typedef unsigned long		u32;
typedef unsigned long long	u64;
#endif // !u8_u16_u32_u64

typedef enum {
	SYSCALL_GROUP_VMM		= 0x00000000,
	SYSCALL_GROUP_TIMER		= 0x01000000,
	SYSCALL_GROUP_DISP		= 0x02000000,
	SYSCALL_GROUP_FS		= 0x03000000,
	SYSCALL_GROUP_MEMORY	= 0x04000000,
	SYSCALL_GROUP_VM		= 0x0F000000,
} SYSCALL_GROUP_ID;

#define SYSCALL_GROUP(func)		((func) & 0xFF000000)

typedef enum {
	// VMM関連
	SYSCALL_OUTPUT_DBGSTR = SYSCALL_GROUP_VMM + 1,

	// タイマ関連
	SYSCALL_GET_TICK_COUNT = SYSCALL_GROUP_TIMER + 1,
	SYSCALL_GET_SYSTEM_TIME,
	SYSCALL_GET_LOCAL_TIME,

	// 表示関連
	SYSCALL_GET_VIDEO_INFO = SYSCALL_GROUP_DISP + 1,
	SYSCALL_PUT_STRING,
	SYSCALL_PUT_CHAR,
	SYSCALL_BIT_BLT,

	// ファイル操作
	SYSCALL_OPEN_FILE = SYSCALL_GROUP_FS + 1,
	SYSCALL_CLOSE_FILE,
	SYSCALL_READ_FILE,
	SYSCALL_SET_FILE_PTR,
	SYSCALL_GET_FILE_SIZE,

	// メモリ操作
	SYSCALL_HEAP_ALLOC = SYSCALL_GROUP_MEMORY + 1,
	SYSCALL_HEAP_FREE,

	// VM操作
	SYSCALL_GET_VCPU = SYSCALL_GROUP_VM + 1,
	SYSCALL_RUN_VCPU,
	SYSCALL_SET_VCPU_EVENT,
	SYSCALL_SET_VCPU_MMIO,
	SYSCALL_SET_VM_EVENTCB,
} SYSCALL_FUNC_ID;

#define syscall_0( result_type, result_asm, func )		\
({														\
	result_type _result;								\
														\
	__asm__ __volatile__(								\
		"int	%1	\n\t"								\
		: "=" result_asm (_result)						\
		: "i"(SYSCALL_VECTOR),							\
		  "a"(func) );									\
														\
	_result;											\
})

#define syscall_1( result_type, result_asm, func, arg1 )	\
({															\
	result_type _result;									\
															\
	__asm__ __volatile__(									\
		"mov	%3, %%ecx	\n\t"							\
		"int	%1			\n\t"							\
		: "=" result_asm (_result)							\
		: "i"(SYSCALL_VECTOR),								\
		  "a"(func), "g"(arg1)								\
		: "%ecx" );											\
															\
	_result;												\
})

#define syscall_2( result_type, result_asm, func, arg1, arg2 )		\
({																	\
	result_type _result;											\
																	\
	__asm__ __volatile__(											\
		"mov	%3, %%ecx	\n\t"									\
		"mov	%4, %%edx	\n\t"									\
		"int	%1			\n\t"									\
		: "=" result_asm (_result)									\
		: "i"(SYSCALL_VECTOR),										\
		  "a"(func), "g"(arg1), "g"(arg2)							\
		: "%ecx", "%edx" );											\
																	\
	_result;														\
})

#define syscall_3( result_type, result_asm, func, arg1, arg2, arg3 )	\
({																		\
	result_type _result;												\
																		\
	__asm__ __volatile__(												\
		"mov	%3, %%ecx	\n\t"										\
		"mov	%4, %%edx	\n\t"										\
		"mov	%5, %%ebx	\n\t"										\
		"int	%1			\n\t"										\
		: "=" result_asm (_result)										\
		: "i"(SYSCALL_VECTOR),											\
		  "a"(func), "g"(arg1), "g"(arg2), "g"(arg3)					\
		: "%ecx", "%edx", "%ebx" );										\
																		\
	_result;															\
})

#define Vsun86_OutputDebugString( s )	\
	syscall_1( int, "a", SYSCALL_OUTPUT_DBGSTR, (s) )

#define Vsun86_GetTickCount()	\
	syscall_0( unsigned long long, "A", SYSCALL_GET_TICK_COUNT )

#define Vsun86_GetSystemTime( p )	\
	syscall_1( int, "a", SYSCALL_GET_SYSTEM_TIME, (p) )

#define Vsun86_GetLocalTime( p )	\
	syscall_1( int, "a", SYSCALL_GET_LOCAL_TIME, (p) )

#define Vsun86_GetVideoInfo( p )	\
	syscall_1( int, "a", SYSCALL_GET_VIDEO_INFO, (p) )

#define Vsun86_PutString( s )	\
	syscall_1( int, "a", SYSCALL_PUT_STRING, (s) )

#define Vsun86_PutChar( c )		\
	syscall_1( int, "a", SYSCALL_PUT_CHAR, (c) )

#define Vsun86_BitBlt( src )	\
	syscall_1( int, "a", SYSCALL_BIT_BLT, (src) )

#define Vsun86_OpenFile( filename )	\
	syscall_1( int, "a", SYSCALL_OPEN_FILE, (filename) )

#define Vsun86_CloseFile( fd )	\
	syscall_1( int, "a", SYSCALL_CLOSE_FILE, (fd) )

#define Vsun86_ReadFile( fd, buf, size )	\
	syscall_3( int, "a", SYSCALL_READ_FILE, (fd), (buf), (size) )

#define Vsun86_SetFilePointer( fd, ptr, method )	\
	syscall_3( int, "a", SYSCALL_SET_FILE_PTR, (fd), (ptr), (method) )

#define Vsun86_GetFileSize( fd, size )	\
	syscall_2( int, "a", SYSCALL_GET_FILE_SIZE, (fd), (size) )

#define Vsun86_HeapAlloc( size )	\
	syscall_1( void *, "a", SYSCALL_HEAP_ALLOC, (size) )

#define Vsun86_HeapFree( ptr )	\
	syscall_1( int, "a", SYSCALL_HEAP_FREE, (ptr) )

#define Vsun86_GetVCPU()	\
	syscall_0( void *, "a", SYSCALL_GET_VCPU )

#define Vsun86_RunVCPU( result )	\
	syscall_1( int, "a", SYSCALL_RUN_VCPU, (result) )

#define Vsun86_SetVCPUEvent( e )	\
	syscall_1( int, "a", SYSCALL_SET_VCPU_EVENT, (e) )

#define Vsun86_SetVCPU_MMIO( start_page, pages, procs )	\
	syscall_3( int, "a", SYSCALL_SET_VCPU_MMIO, (start_page), (pages), (procs) )

#define Vsun86_SetVMEventCallback( proc )		\
	syscall_1( int, "a", SYSCALL_SET_VM_EVENTCB, (proc) )

// for Vsun86_OpenFile()
#define FILE_DESC_INVALID	(-1)

#pragma pack(1)

// for Vsun86_GetSystemTime() / Vsun86_GetLocalTime()
typedef struct {
	u16		year;
	u8		mon;
	u8		mday;
	u8		wday;
	u8		hour;
	u8		min;
	u8		sec;
	u16		msec;
	u16		yday;
} VSUN86_SYSTEM_TIME;

// for Vsun86_GetVideoInfo()
typedef struct {
	u16		width;
	u16		height;
	u8		bpp;
	u8		rsvd[3];
	void *	vram;
} VSUN86_VIDEO_INFO;

// for Vsun86_RunVCPU()
typedef enum {
	VCPU_RESULT_ABORT = -1,
	VCPU_RESULT_NO_ERROR = 0,
	VCPU_RESULT_ICEBP,
	VCPU_RESULT_INB,
	VCPU_RESULT_INW,
	VCPU_RESULT_IND,
	VCPU_RESULT_OUTB,
	VCPU_RESULT_OUTW,
	VCPU_RESULT_OUTD,
} VCPU_RESULT_CODE;

typedef struct {
	VCPU_RESULT_CODE	code;
	union {
		u32			d[3];
		struct {
			u16		port;
			u8		str;
			u8		rep;
			u32		next_eip;
			u32		rsvd;
		} ioio;
	} data;
} VSUN86_VCPU_RESULT;

// for Vsun86_SetVCPUEvent()
typedef enum {
	VCPU_EVENT_INTR,
	VCPU_EVENT_NMI,
	VCPU_EVENT_EXCEPTION,
	VCPU_EVENT_SWINT,
} VCPU_EVENT_CODE;

typedef struct {
	VCPU_EVENT_CODE		code;
	struct {
		u8		vector;
		u8		error_code_valid;
		u16		rsvd;
		u32		error_code;
	} data;
} VSUN86_VCPU_EVENT;

// for Vsun86_SetEventCallback()
typedef enum {
	VM_EVENT_NOP,
	VM_EVENT_KEY_DOWN,
	VM_EVENT_KEY_UP,
} VM_EVENT_CODE;

typedef int (CALLBACK *VM_EVENT_CALLBACK)( VM_EVENT_CODE code, u32 arg1, u32 arg2 );

#pragma pack()

#endif // !__SYSCALL_H__
