/*!
******************************************************************************

	@file	vcpu.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "cpu.h"
#include "pfemu.h"
#include "pfemu/vcpu.h"
#include "pfemu/softemu.h"
#include "msr.h"
#include "timer.h"
#include "printf.h"
#include "syscall.h"
#include "pic.h"
#include "lapic.h"
#include "io.h"

bool vcpu_init( VSUN86_VM *vm )
{
	VCPU *cpu = (VCPU *)vm->vcpu;
	SVM_VMCB *vmcb = (SVM_VMCB *)vm->vmcb;

	memset( cpu, 0, sizeof(VCPU) );
	cpu->vmcb = vmcb;

	// VMCBを設定する
	vmcb->ctrl.intercept_cr_read  = 0xFFFF;
	vmcb->ctrl.intercept_cr_write = 0xFFFF;
	vmcb->ctrl.intercept_dr_read  = 0xFFFF;
	vmcb->ctrl.intercept_dr_write = 0xFFFF;
	vmcb->ctrl.intercept_excp	  = 0xFFFFFFFF;
	vmcb->ctrl.intercept_mask	  = ~( SVM_INTERCEPT_RESERVED
									 | SVM_INTERCEPT_IDTR_READ
									 | SVM_INTERCEPT_GDTR_READ
									 | SVM_INTERCEPT_LDTR_READ
									 | SVM_INTERCEPT_TR_READ
									 | SVM_INTERCEPT_IDTR_WRITE
									 | SVM_INTERCEPT_GDTR_WRITE
									 | SVM_INTERCEPT_LDTR_WRITE
									 | SVM_INTERCEPT_TR_WRITE
									 | SVM_INTERCEPT_PUSHF
									 | SVM_INTERCEPT_POPF
									 | SVM_INTERCEPT_INT
									 | SVM_INTERCEPT_IRET
									);
	vmcb->ctrl.iopm_base_pa		  = virt_to_phys( vm->iopm  );
	vmcb->ctrl.msrpm_base_pa	  = virt_to_phys( vm->msrpm );
	vmcb->ctrl.asid				  = vm->vmid + 1;
	vmcb->ctrl.tlb_control		  = 1;	// ★暫定(正常動作が確認できたら0にする！)
	vmcb->ctrl.vint_flags		  = SVM_V_INTR_MASKING;

	// 仮想CPUのレジスタを初期化する
	_EFLAGS(cpu)		= 0x00000002;
	_EIP(cpu)			= 0x0000FFF0;
	V_CR0(cpu).q		= CR0_CD | CR0_NW | CR0_ET;
	R_CR0(cpu).q		= CR0_PG | CR0_WP | V_CR0(cpu).q;	// Paged Real Modeを利用
	R_CR3(cpu).q		= virt_to_phys( vm->page_dir );
	_EFER(cpu)			= EFER_SVME;
	_CS(cpu).selector	= 0xF000;
	_CS(cpu).base		= 0x000F0000;	// ホントは「0xFFFF0000」だけど…
	_CS(cpu).limit		= 0xFFFF;
	_CS(cpu).attrib		= SVM_VMCB_SEG_ATTRIB_CS( 0, 0, CS_READABLE );
	_SS(cpu).limit		= 0xFFFF;
	_SS(cpu).attrib		= SVM_VMCB_SEG_ATTRIB_DS( 0, 0, DS_WRITABLE );
	_DS(cpu).limit		= 0xFFFF;
	_DS(cpu).attrib		= SVM_VMCB_SEG_ATTRIB_DS( 0, 0, DS_WRITABLE );
	_ES(cpu).limit		= 0xFFFF;
	_ES(cpu).attrib		= SVM_VMCB_SEG_ATTRIB_DS( 0, 0, DS_WRITABLE );
	_FS(cpu).limit		= 0xFFFF;
	_FS(cpu).attrib		= SVM_VMCB_SEG_ATTRIB_DS( 0, 0, DS_WRITABLE );
	_GS(cpu).limit		= 0xFFFF;
	_GS(cpu).attrib		= SVM_VMCB_SEG_ATTRIB_DS( 0, 0, DS_WRITABLE );
	_EDX(cpu)			= 0x00000600;	// ホストのCPUIDを取得すべきかも？
	_GDTR(cpu).limit	= 0xFFFF;
	_IDTR(cpu).limit	= 0xFFFF;
	_LDTR(cpu).limit	= 0xFFFF;
	_LDTR(cpu).attrib	= SVM_VMCB_SEG_ATTRIB_DEFAULT_LDTR;
	_TR(cpu).limit		= 0xFFFF;
	_TR(cpu).attrib		= SVM_VMCB_SEG_ATTRIB_DEFAULT_TR;
	_DR6(cpu).q			= 0xFFFF0FF0;
	_DR7(cpu).q			= 0x00000400;

	// A20Mを有効にする
	vcpu_enable_a20m( vm );

	return true;
}

u32 guest_ecx, guest_edx, guest_ebx;
u32 guest_ebp, guest_esi, guest_edi;

bool vcpu_run( VSUN86_VM *vm, void *p )
{
	VSUN86_VCPU_RESULT *result = (VSUN86_VCPU_RESULT *)p;
	if ( result == NULL )
		return false;

	register VCPU *cpu = (VCPU *)vm->vcpu;
	u32 eflags;
	__ASM__( "pushfl		\n\t"
			 "popl		%0	\n\t"
			 : "=m"(eflags) );

	BASELIMIT gdt_ptr;
	DESCRIPTOR *tss_desc;
	u32 host_cr2, host_fs, host_gs;
	u16 host_ldtr, host_tr;
	u32 host_dr0, host_dr1, host_dr2, host_dr3, host_dr6, host_dr7;
	u64 kernel_gs_base, star, lstar, cstar, fmask;
	u64 sysenter_cs, sysenter_esp, sysenter_eip;
//	u8  host_fx_env[0x200];
	u64 tsc_start, tsc_end;

	const u8 imr0 = pic_get_imr( PIC_MASTER );
	const u8 imr1 = pic_get_imr( PIC_SLAVE  );
	outb( IO_PIC0_OCW1, 0xFE );
	outb( IO_PIC1_OCW1, 0xFF );
	cpu_enable_interrupt();
	while ( pic_get_isr( PIC_MASTER ) );
	cpu_disable_interrupt();
	lapic_disable_lint0();

	// デバッグレジスタを切り替える
	X86_GET_DR( 0, host_dr0 );
	X86_GET_DR( 1, host_dr1 );
	X86_GET_DR( 2, host_dr2 );
	X86_GET_DR( 3, host_dr3 );
	X86_GET_DR( 6, host_dr6 );
	X86_GET_DR( 7, host_dr7 );
	X86_SET_DR( 7, 0 );	// DR0～DR3をセットした時に変なところでブレークしないように…
	X86_SET_DR( 0, _DR0(cpu).d );
	X86_SET_DR( 1, _DR1(cpu).d );
	X86_SET_DR( 2, _DR2(cpu).d );
	X86_SET_DR( 3, _DR3(cpu).d );

	/*
	// X87/XMMレジスタを切り替える
	X86_FXSAVE( host_fx_env );
	X86_FXRSTOR( cpu->fx_env );
	*/

	// CR2, FS, GS, LDTR, TRを保存する
	X86_GET_CR( 2, host_cr2);
	X86_GET_FS( host_fs );
	X86_GET_GS( host_gs );
	X86_SLDT( host_ldtr );
	X86_STR ( host_tr   );
	X86_SGDT( gdt_ptr   );
	tss_desc = (DESCRIPTOR *)( gdt_ptr.base + (host_tr & 0xFFF8) );

	// VMLOADで上書きされるMSRを保存する
	kernel_gs_base	= rdmsr( MSR_KERNEL_GS_BASE );
	star			= rdmsr( MSR_STAR );
	lstar			= rdmsr( MSR_LSTAR );
	cstar			= rdmsr( MSR_CSTAR );
	fmask			= rdmsr( MSR_FMASK );
	sysenter_cs		= rdmsr( MSR_SYSENTER_CS );
	sysenter_esp	= rdmsr( MSR_SYSENTER_ESP );
	sysenter_eip	= rdmsr( MSR_SYSENTER_EIP );

	// 実行開始時間を保存する
	tsc_start = rdtsc();

	// VM環境を動かす
	/*
	__ASM__(
		"clgi							\n\t"
		"sti							\n\t"
		"pushal							\n\t"
		"movl	%c[rcx](%[cpu]), %%ecx	\n\t"
		"movl	%c[rdx](%[cpu]), %%edx	\n\t"
		"movl	%c[rbx](%[cpu]), %%ebx	\n\t"
		"movl	%c[rbp](%[cpu]), %%ebp	\n\t"
		"movl	%c[rsi](%[cpu]), %%esi	\n\t"
		"movl	%c[rdi](%[cpu]), %%edi	\n\t"
		"pushl	%%eax					\n\t"
		"movl	%c[vmcb](%[cpu]), %%eax	\n\t"
		"vmload							\n\t"
		"vmrun							\n\t"
		"vmsave							\n\t"
		"popl	%%eax					\n\t"
		"movl	%%ecx, %c[rcx](%[cpu])	\n\t"
		"movl	%%edx, %c[rdx](%[cpu])	\n\t"
		"movl	%%ebx, %c[rbx](%[cpu])	\n\t"
		"movl	%%ebp, %c[rbp](%[cpu])	\n\t"
		"movl	%%esi, %c[rsi](%[cpu])	\n\t"
		"movl	%%edi, %c[rdi](%[cpu])	\n\t"
		"popal							\n\t"
		"cli							\n\t"
		"stgi							\n\t"
		: [cpu] "+a"(cpu)
		: [rcx] "g"(offsetof(VCPU, rcx)),
		  [rdx] "g"(offsetof(VCPU, rdx)),
		  [rbx] "g"(offsetof(VCPU, rbx)),
		  [rbp] "g"(offsetof(VCPU, rbp)),
		  [rsi] "g"(offsetof(VCPU, rsi)),
		  [rdi] "g"(offsetof(VCPU, rdi)),
		  [vmcb]"g"(offsetof(VCPU, vmcb))
		: "memory", "cc"
	);
	*/
	guest_ecx = cpu->rcx.d;
	guest_edx = cpu->rdx.d;
	guest_ebx = cpu->rbx.d;
	guest_ebp = cpu->rbp.d;
	guest_esi = cpu->rsi.d;
	guest_edi = cpu->rdi.d;
	__ASM__(
		"clgi							\n\t"
		"sti							\n\t"
		"pushal							\n\t"
		"movl	guest_ecx, %%ecx		\n\t"
		"movl	guest_edx, %%edx		\n\t"
		"movl	guest_ebx, %%ebx		\n\t"
		"movl	guest_ebp, %%ebp		\n\t"
		"movl	guest_esi, %%esi		\n\t"
		"movl	guest_edi, %%edi		\n\t"
		"vmload							\n\t"
		"vmrun							\n\t"
		"vmsave							\n\t"
		"movl	%%ecx, guest_ecx		\n\t"
		"movl	%%edx, guest_edx		\n\t"
		"movl	%%ebx, guest_ebx		\n\t"
		"movl	%%ebp, guest_ebp		\n\t"
		"movl	%%esi, guest_esi		\n\t"
		"movl	%%edi, guest_edi		\n\t"
		"popal							\n\t"
		"cli							\n\t"
		"stgi							\n\t"
		: [vmcb] "+a"(cpu->vmcb)
		:: "memory", "cc"
	);
	cpu->rcx.d = guest_ecx;
	cpu->rdx.d = guest_edx;
	cpu->rbx.d = guest_ebx;
	cpu->rbp.d = guest_ebp;
	cpu->rsi.d = guest_esi;
	cpu->rdi.d = guest_edi;

	// 実行終了時間を保存する
	tsc_end = rdtsc();

	// VMLOADで上書きされたMSRを復帰する
	wrmsr( MSR_KERNEL_GS_BASE,	kernel_gs_base );
	wrmsr( MSR_STAR,			star  );
	wrmsr( MSR_LSTAR,			lstar );
	wrmsr( MSR_CSTAR,			cstar );
	wrmsr( MSR_FMASK,			fmask );
	wrmsr( MSR_SYSENTER_CS,		sysenter_cs  );
	wrmsr( MSR_SYSENTER_ESP,	sysenter_esp );
	wrmsr( MSR_SYSENTER_EIP,	sysenter_eip );

	// CR2, FS, GS, LDTR, TRを復帰する
	X86_SET_CR( 2, host_cr2 );
	X86_SET_FS( host_fs );
	X86_SET_GS( host_gs );
	X86_LLDT( host_ldtr );
	tss_desc->seg.type &= ~DESC_TSS_BUSY;	// BUSY=0
	X86_LTR( host_tr );

	/*
	// X87/XMMレジスタを切り替える
	X86_FXSAVE( cpu->fx_env );
	X86_FXRSTOR( host_fx_env );
	*/

	// デバッグレジスタを切り替える
	X86_GET_DR( 0, _DR0(cpu).d );
	X86_GET_DR( 1, _DR1(cpu).d );
	X86_GET_DR( 2, _DR2(cpu).d );
	X86_GET_DR( 3, _DR3(cpu).d );
	X86_SET_DR( 0, host_dr0 );
	X86_SET_DR( 1, host_dr1 );
	X86_SET_DR( 2, host_dr2 );
	X86_SET_DR( 3, host_dr3 );
	X86_SET_DR( 6, host_dr6 );
	X86_SET_DR( 7, host_dr7 );

	outb( IO_PIC0_OCW1, imr0 );
	outb( IO_PIC1_OCW1, imr1 );
	lapic_enable_lint0();

	memset( result, 0, sizeof(VSUN86_VCPU_RESULT) );

	u8 *mem = (u8 *)vm->mem;
	switch ( SVM_EXIT_CODE( cpu->vmcb ) )
	{
	case SVM_VMEXIT_SHUTDOWN:	// シャットダウン
		result->code = VCPU_RESULT_ABORT;
		vmm_printf( VMM_ERROR, "VM shutdown.\n" );
		break;

	case SVM_VMEXIT_INTR:
		{	// ハードウェア割り込み
			result->code = VCPU_RESULT_NO_ERROR;
			return true;
		}
		break;

	case SVM_VMEXIT_READ_CR(0):
	case SVM_VMEXIT_READ_CR(2):
	case SVM_VMEXIT_READ_CR(3):
	case SVM_VMEXIT_READ_CR(4):
		if ( vcpu_intercept_read_cr( cpu, mem ) ) {
			result->code = VCPU_RESULT_NO_ERROR;
			return true;
		}
		result->code = VCPU_RESULT_ABORT;
		break;

	case SVM_VMEXIT_EXCP(14):
		{	// #PF -> MMIOエミュレーション
			const u32 exit_code1 = (u32)SVM_EXIT_INFO1( cpu->vmcb );	// エラーコード
			const u32 exit_code2 = (u32)SVM_EXIT_INFO2( cpu->vmcb );	// #PFアドレス
			const u32 page = exit_code2 >> 12;
			if ( (0xA0 <= page) || (page <= 0xFF) ) {
				if ( vcpu_emulate( cpu, mem ) ) {
					result->code = VCPU_RESULT_NO_ERROR;
					return true;
				}
			}
			result->code = VCPU_RESULT_ABORT;
			vmm_printf( VMM_DEBUG, "SVM_VMEXIT_EXCP(14) !!!\n" );
			vmm_printf( VMM_DEBUG, "EXITINFO1 ... %08x\n", exit_code1 );
			vmm_printf( VMM_DEBUG, "EXITINFO2 ... %08x\n", exit_code2 );
		}
		break;

	case SVM_VMEXIT_IOIO:
		{	// ポートI/O命令
			const u32 exit_info1 = (u32)SVM_EXIT_INFO1( cpu->vmcb );
			if ( exit_info1 & 0x200 ) {		// A64
				result->code = VCPU_RESULT_ABORT;	// 64ビットアドレスモードは非対応
				break;
			}
			if ( exit_info1 & 0x01 )
			{	// TYPE=IN
				if      ( exit_info1 & 0x10 ) result->code = VCPU_RESULT_INB;	// SZ8
				else if ( exit_info1 & 0x20 ) result->code = VCPU_RESULT_INW;	// SZ16
				else if ( exit_info1 & 0x40 ) result->code = VCPU_RESULT_IND;	// SZ32
			}
			else
			{	// TYPE=OUT
				if      ( exit_info1 & 0x10 ) result->code = VCPU_RESULT_OUTB;	// SZ8
				else if ( exit_info1 & 0x20 ) result->code = VCPU_RESULT_OUTW;	// SZ16
				else if ( exit_info1 & 0x40 ) result->code = VCPU_RESULT_OUTD;	// SZ32
			}
			if ( exit_info1 & 0x04 ) result->data.ioio.str = 1;		// STR
			if ( exit_info1 & 0x08 ) result->data.ioio.rep = 1;		// REP
			result->data.ioio.port = (u16)(exit_info1 >> 16);		// PORT
			result->data.ioio.next_eip = (u32)SVM_EXIT_INFO2( cpu->vmcb );
		}
		break;

	case SVM_VMEXIT_ICEBP:
		{	// ICEBP命令
			result->code = VCPU_RESULT_ICEBP;
		}
		break;

	default:
		result->code = VCPU_RESULT_ABORT;
		vmm_printf( VMM_DEBUG, "EXITCODE ...... %016llx\n", SVM_EXIT_CODE( cpu->vmcb ) );
		vmm_printf( VMM_DEBUG, "EXITINFO1 ..... %016llx\n", SVM_EXIT_INFO1( cpu->vmcb ) );
		vmm_printf( VMM_DEBUG, "EXITINFO2 ..... %016llx\n", SVM_EXIT_INFO2( cpu->vmcb ) );
		vmm_printf( VMM_DEBUG, "EXITINTINFO ... %016llx\n", SVM_EXIT_INT_INFO( cpu->vmcb ) );
		break;
	}
	return false;
}

bool vcpu_set_event( VSUN86_VM *vm, void *p )
{
	VSUN86_VCPU_EVENT *e = (VSUN86_VCPU_EVENT *)p;
	if ( e == NULL )
		return false;

	SVM_VMCB *vmcb = ((VCPU *)vm->vcpu)->vmcb;

	switch ( e->code )
	{
	case VCPU_EVENT_INTR:
		vmcb->ctrl.event_inj = 0x80000000 | (0<<8) | e->data.vector;
		break;

	case VCPU_EVENT_NMI:
		vmcb->ctrl.event_inj = 0x80000000 | (2<<8);
		break;

	case VCPU_EVENT_EXCEPTION:
		vmcb->ctrl.event_inj = 0x80000000 | (3<<8) | e->data.vector;
		if ( e->data.error_code_valid )
			vmcb->ctrl.event_inj |= 0x800 | ((u64)e->data.error_code << 32);
		break;

	case VCPU_EVENT_SWINT:
		vmcb->ctrl.event_inj = 0x80000000 | (4<<8) | e->data.vector;
		break;

	default:
		vmcb->ctrl.event_inj = 0;
		return false;
	}

	return true;
}

bool vcpu_set_mmio( VSUN86_VM *vm, u32 start_page, u32 pages, void *p )
{
	VCPU_MMIO_PROCS *procs = (VCPU_MMIO_PROCS *)p;

	if ( p == NULL )
		return false;

	if ( (start_page < 0xA0) || (0xFF < start_page+pages) )
		return false;

	VCPU *cpu = (VCPU *)vm->vcpu;
	VCPU_MMIO_PROCS *mmio = &cpu->mmio[start_page-0xA0];
	u32 *page_tbl = (u32 *)vm->page_tbl;

	vmm_printf( VMM_DEBUG, "vcpu_set_mmio(): %08x %08x %08x\n", cpu, mmio, page_tbl );

	for ( u32 i=0; i<pages; i++ ) {
		mmio[i].read8	= procs->read8;
		mmio[i].read16	= procs->read16;
		mmio[i].read32	= procs->read32;
		mmio[i].write8	= procs->write8;
		mmio[i].write16	= procs->write16;
		mmio[i].write32	= procs->write32;
		if ( !mmio[i].read8 && !mmio[i].read16 && !mmio[i].read32 )
		{	// 直接メモリからリードするのでライトプロテクトのみ
			page_tbl[start_page+i] |= PTE_PRESENT;		// P=1
			page_tbl[start_page+i] &= ~PTE_READWRITE;	// R/W=0
		}
		else
		{	// リード側も捕捉するためにページを無効にする
			page_tbl[start_page+i] &= ~PTE_PRESENT;		// P=0
		}
	}

	return true;
}

VCPU_MMIO_PROCS * vcpu_get_mmio( VCPU *cpu, u32 page )
{
	static VCPU_MMIO_PROCS dummy_mmio = { NULL, NULL, NULL, NULL, NULL, NULL };
	if ( (0xA0 <= page) && (page <= 0xFF) )
		return &cpu->mmio[page-0xA0];

	return &dummy_mmio;
}

SVM_VMCB_SEG * vcpu_get_seg( VCPU *cpu, u8 index )
{
	switch ( index )
	{
	case VCPU_SEG_INDEX_NULL:	return &_DS(cpu);
	case VCPU_SEG_INDEX_ES:		return &_ES(cpu);
	case VCPU_SEG_INDEX_CS:		return &_CS(cpu);
	case VCPU_SEG_INDEX_SS:		return &_SS(cpu);
	case VCPU_SEG_INDEX_DS:		return &_DS(cpu);
	case VCPU_SEG_INDEX_FS:		return &_FS(cpu);
	case VCPU_SEG_INDEX_GS:		return &_GS(cpu);
	}

	return NULL;
}

bool vcpu_enable_a20m( VSUN86_VM *vm )
{
	VCPU *cpu = (VCPU *)vm->vcpu;
	if ( _A20M(cpu) )
		return true;

	u32 *page_tbl = (u32 *)vm->page_tbl;
	for ( u8 i=0; i<16; i++ ) {
		cpu->a20_pte[i]   = page_tbl[0x100+i];
		page_tbl[0x100+i] = page_tbl[0x000+i];
	}
	_A20M(cpu) = 1;

	return true;
}

bool vcpu_disable_a20m( VSUN86_VM *vm )
{
	VCPU *cpu = (VCPU *)vm->vcpu;
	if ( !_A20M(cpu) )
		return true;

	u32 *page_tbl = (u32 *)vm->page_tbl;
	for ( u8 i=0; i<16; i++ )
		page_tbl[0x100+i] = cpu->a20_pte[i];
	_A20M(cpu) = 0;

	return true;
}
