/*!
******************************************************************************

	@file	softemu.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "cpu.h"
#include "pfemu.h"
#include "pfemu/vcpu.h"
#include "pfemu/softemu.h"
#include "printf.h"

typedef SVM_REG * ( *GET_REG )( VCPU * );
typedef SVM_VMCB_SEG * ( *GET_SEG )( VCPU * );

static SVM_REG * get_rax( VCPU *cpu ) { return &_rAX(cpu); }
static SVM_REG * get_rcx( VCPU *cpu ) { return &_rCX(cpu); }
static SVM_REG * get_rdx( VCPU *cpu ) { return &_rDX(cpu); }
static SVM_REG * get_rbx( VCPU *cpu ) { return &_rBX(cpu); }
static SVM_REG * get_rsp( VCPU *cpu ) { return &_rSP(cpu); }
static SVM_REG * get_rbp( VCPU *cpu ) { return &_rBP(cpu); }
static SVM_REG * get_rsi( VCPU *cpu ) { return &_rSI(cpu); }
static SVM_REG * get_rdi( VCPU *cpu ) { return &_rDI(cpu); }

static GET_REG gp_reg[8] = {
	get_rax,	get_rcx,	get_rdx,	get_rbx,
	get_rsp,	get_rbp,	get_rsi,	get_rdi
};

static SVM_REG * get_cr0( VCPU *cpu ) { return &V_CR0(cpu); }
static SVM_REG * get_cr2( VCPU *cpu ) { return &V_CR2(cpu); }
static SVM_REG * get_cr3( VCPU *cpu ) { return &V_CR3(cpu); }
static SVM_REG * get_cr4( VCPU *cpu ) { return &V_CR4(cpu); }

static GET_REG ctrl_reg[8] = {
	get_cr0,	NULL,		get_cr2,	get_cr3,
	get_cr4,	NULL,		NULL,		NULL
};

bool vcpu_emulate( VCPU *cpu, u8 *mem )
{
	if ( V_CR0(cpu).d & CR0_PG )
		return false;	// ページング有効時は未実装

	u32 op_info = 0;

	u8 op_code;
	u8 n = 0;
	while ( n < 16 ) {
		op_code = mem[_CS(cpu).base + _EIP(cpu) + n];
		switch ( op_code ) {
		case 0xF0:	// LOCK
			n++;
			continue;
		case 0x26:	// ES:
			op_info &= ~OP_PREFIX_SEG_MASK;
			op_info |= OP_PREFIX_SEG_ES;
			n++;
			continue;
		case 0x2E:	// CS:
			op_info &= ~OP_PREFIX_SEG_MASK;
			op_info |= OP_PREFIX_SEG_CS;
			n++;
			continue;
		case 0x36:	// SS:
			op_info &= ~OP_PREFIX_SEG_MASK;
			op_info |= OP_PREFIX_SEG_SS;
			n++;
			continue;
		case 0x3E:	// DS:
			op_info &= ~OP_PREFIX_SEG_MASK;
			op_info |= OP_PREFIX_SEG_DS;
			n++;
			continue;
		case 0x64:	// FS:
			op_info &= ~OP_PREFIX_SEG_MASK;
			op_info |= OP_PREFIX_SEG_FS;
			n++;
			continue;
		case 0x65:	// GS:
			op_info &= ~OP_PREFIX_SEG_MASK;
			op_info |= OP_PREFIX_SEG_GS;
			n++;
			continue;
		case 0x66:	// Operand Size
			if ( !SVM_VMCB_SEG_SIZE_IS_32BIT( _CS(cpu) ) )
				op_info |= OP_PREFIX_CODE32;
			n++;
			continue;
		case 0x67:	// Address Size
			if ( !SVM_VMCB_SEG_SIZE_IS_32BIT( _CS(cpu) ) )
				op_info |= OP_PREFIX_ADDR32;
			n++;
			continue;
		case 0xF2:	// REPNE
			op_info |= OP_PREFIX_REPNE;
			n++;
			continue;
		case 0xF3:	// REP/REPE
			op_info |= OP_PREFIX_REPE;
			n++;
			continue;
		}
		break;
	}
	if ( n >= 16 )
		return false;

	if ( op_code == 0x0F ) {
		n++;
		op_code = mem[_CS(cpu).base + _EIP(cpu) + n];
		op_info |= (op_code << OP_CODE_SHIFT) | OP_PREFIX_2BYTES | (n+1);
		vmm_printf( VMM_DEBUG, "vcpu_emulate(): unsupported instruction( %08x )\n", op_info );
	}
	else {
		op_info |= (op_code << OP_CODE_SHIFT) | (n+1);
		bool emu_ok = false;
		switch ( op_code )
		{
		case 0x86:	// XCHG r/m8, r8
			emu_ok = vcpu_emulate_xchg( cpu, mem, op_info | OP_USE_MODRM | OP_USE_8BIT_REG );
			break;

		case 0x87:	// XCHG r/m, reg
			emu_ok = vcpu_emulate_xchg( cpu, mem, op_info | OP_USE_MODRM );
			break;

		case 0x88:	// MOV r/m8, r8
		case 0x8A:	// MOV r8, r/m8
			emu_ok = vcpu_emulate_mov_modrm( cpu, mem, op_info | OP_USE_MODRM | OP_USE_8BIT_REG );
			break;

		case 0x89:	// MOV r/m, reg
		case 0x8B:	// MOV reg, r/m
			emu_ok = vcpu_emulate_mov_modrm( cpu, mem, op_info | OP_USE_MODRM );
			break;

		case 0xA0:	// MOV AL, moffs8
		case 0xA1:	// MOV rAX, moffs
		case 0xA2:	// MOV moffs8, AL
		case 0xA3:	// MOV moffs, rAX
			emu_ok = vcpu_emulate_mov_moffs( cpu, mem, op_info );
			break;

		case 0xA4:	// MOVSB
		case 0xA5:	// MOVSW/MOVSD
			emu_ok = vcpu_emulate_movs( cpu, mem, op_info );
			break;

		case 0xAA:	// STOSB
		case 0xAB:	// STOSW/STOSD
			emu_ok = vcpu_emulate_stos( cpu, mem, op_info );
			break;

		default:
			break;
		}
		if ( emu_ok )
			return true;

		vmm_printf( VMM_DEBUG, "vcpu_emulate(): emulation failed. ( op_info=%08x )\n", op_info );
	}

	return false;
}

bool vcpu_intercept_read_cr( VCPU *cpu, u8 *mem )
{
	u8 modrm, n;

	if ( V_CR0(cpu).d & CR0_PG )
		return false;	// ページング有効時は未実装

	// 「0F xx」までスキップ
	while ( mem[_CS(cpu).base+_EIP(cpu)] != 0x0F )
		_EIP(cpu)++;
	_EIP(cpu)++;

	if ( mem[_CS(cpu).base+_EIP(cpu)] == 0x20 )
	{	// [0F 20 /r] : mov reg, cr*
		_EIP(cpu)++;

		modrm = mem[_CS(cpu).base+_EIP(cpu)];
		if ( modrm < 0xC0 )
			return false;

		n = (modrm >> 3) & 0x07;
		if ( ctrl_reg[n] == NULL )
			return false;
		gp_reg[modrm & 0x07](cpu)->d = ctrl_reg[n](cpu)->d;
		_EIP(cpu)++;
		return true;
	}
	else if ( mem[_CS(cpu).base+_EIP(cpu)] == 0x01)
	{	// [0F 01 /4] : smsw
		_EIP(cpu)++;

		modrm = mem[_CS(cpu).base+_EIP(cpu)];
		if ( (modrm & 0x38) != 0x20 )
		{	// SMSWじゃない(想定外)
			return false;
		}
		if ( modrm < 0xC0 )
		{	// ★未実装…ホントはメモリアドレスを計算しないと…
			return false;
		}
		// 下位16ビットだけコピー(で良かった？)
		gp_reg[modrm & 0x07](cpu)->w = ctrl_reg[0](cpu)->w;
		_EIP(cpu)++;
		return true;
	}

	return false;
}

bool vcpu_analyze_modrm( VCPU *cpu, u8 *mem, u32 op_info, MODRM_INFO *modrm_info )
{
	if ( V_CR0(cpu).d & CR0_PG )
		return false;	// ページング有効時は未実装

	if ( !(op_info & OP_USE_MODRM) )
		return false;

	const u32 addr = _CS(cpu).base + _EIP(cpu) + OP_BYTES(op_info);
	const u8 modrm = mem[addr];

	modrm_info->reg_index = (modrm >> 3) & 0x07;
	if ( op_info & OP_USE_8BIT_REG ) {
		if ( modrm_info->reg_index < 4 )
			modrm_info->reg = &(gp_reg[modrm_info->reg_index](cpu)->bl);
		else
			modrm_info->reg = &(gp_reg[modrm_info->reg_index-4](cpu)->bh);
	}
	else
		modrm_info->reg = gp_reg[modrm_info->reg_index](cpu);

	if ( 0xC0 == (modrm & 0xC0) ) {
		modrm_info->rm_index = modrm & 0x07;
		if ( op_info & OP_USE_8BIT_REG ) {
			if ( modrm_info->rm_index < 4 )
				modrm_info->rm_reg = &(gp_reg[modrm_info->rm_index](cpu)->bl);
			else
				modrm_info->rm_reg = &(gp_reg[modrm_info->rm_index-4](cpu)->bh);
		}
		else
			modrm_info->rm_reg = gp_reg[modrm_info->rm_index](cpu);
		modrm_info->rm_seg = NULL;
		modrm_info->rm_off = 0;
		modrm_info->bytes  = 1;
		return true;
	}

	modrm_info->rm_index = MODRM_RM_MEM;
	modrm_info->rm_reg	 = NULL;
	modrm_info->rm_seg	 = vcpu_get_seg( cpu, OP_PREFIX_SEG(op_info) >> OP_PREFIX_SEG_SHIFT );
	if ( modrm_info->rm_seg == NULL )
		return false;

	if ( !(op_info & OP_PREFIX_ADDR32) )
	{	// 16ビットアドレス
		switch ( modrm & 0x07 ) {
		case 0x00:	// [BX+SI]
			modrm_info->rm_off = _BX(cpu) + _SI(cpu);
			break;

		case 0x01:	// [BX+DI]
			modrm_info->rm_off = _BX(cpu) + _DI(cpu);
			break;

		case 0x02:	// [BP+SI]
			if ( OP_PREFIX_SEG(op_info) == OP_PREFIX_SEG_NONE )
				modrm_info->rm_seg = vcpu_get_seg( cpu, VCPU_SEG_INDEX_SS );
			modrm_info->rm_off = _BP(cpu) + _SI(cpu);
			break;

		case 0x03:	// [BP+SI]
			if ( OP_PREFIX_SEG(op_info) == OP_PREFIX_SEG_NONE )
				modrm_info->rm_seg = vcpu_get_seg( cpu, VCPU_SEG_INDEX_SS );
			modrm_info->rm_off = _BP(cpu) + _DI(cpu);
			break;

		case 0x04:	// [SI]
			modrm_info->rm_off = _SI(cpu);
			break;

		case 0x05:	// [DI]
			modrm_info->rm_off = _DI(cpu);
			break;

		case 0x06:	// [BP] or disp16
			if ( 0x00 == (modrm & 0xC0) ) {
				modrm_info->rm_off = *(u16 *)&mem[addr+1];
				modrm_info->bytes  = 3;
				return true;
			}
			if ( OP_PREFIX_SEG(op_info) == OP_PREFIX_SEG_NONE )
				modrm_info->rm_seg = vcpu_get_seg( cpu, VCPU_SEG_INDEX_SS );
			modrm_info->rm_off = _BP(cpu);
			break;

		case 0x07:	// [BX]
			modrm_info->rm_off = _BX(cpu);
			break;
		}

		switch ( modrm & 0xC0 )
		{
		case 0x00:
			modrm_info->bytes = 1;
			break;

		case 0x40:	// +imm8
			modrm_info->rm_off += *(char *)&mem[addr+1];
			modrm_info->bytes = 2;
			break;

		case 0x80:	// +imm16
			modrm_info->rm_off += *(u16 *)&mem[addr+1];
			modrm_info->bytes = 3;
			break;
		}
	}
	else
	{	// 32ビットアドレス
		return false;	// ★未実装
	}

	return true;
}
