/*!
******************************************************************************

	@file	lapic.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "cpu.h"
#include "lapic.h"
#include "msr.h"
#include "printf.h"
#include "io.h"

#define LAPIC_ID			0x020
#define LAPIC_VER			0x030
#define LAPIC_TPR			0x080
#define LAPIC_APR			0x090
#define LAPIC_PPR			0x0A0
#define LAPIC_EOI			0x0B0
#define LAPIC_RRD			0x0C0
#define LAPIC_LDR			0x0D0
#define LAPIC_DFR			0x0E0
#define LAPIC_SPINTVEC		0x0F0
#define LAPIC_ISR			0x100
#define LAPIC_TMR			0x180
#define LAPIC_IRR			0x200
#define LAPIC_ESR			0x280
#define LAPIC_INTCMD_L		0x300
#define LAPIC_INTCMD_H		0x310
#define LAPIC_LVT_TIMER		0x320
#define LAPIC_LVT_THERM		0x330
#define LAPIC_LVT_PCNTR		0x340
#define LAPIC_LVT_LINT0		0x350
#define LAPIC_LVT_LINT1		0x360
#define LAPIC_LVT_ERROR		0x370
#define LAPIC_TIMER_ICR		0x380
#define LAPIC_TIMER_CCR		0x390
#define LAPIC_TIMER_DCR		0x3E0
#define LAPIC_EXT_FEATURE	0x400
#define LAPIC_EXT_CTRL		0x410
#define LAPIC_SEOI			0x420
#define LAPIC_IER			0x480
#define LAPIC_LVT_EXINT		0x500

#define LVT_VEC_MASK		0x000000FF
#define LVT_MT_MASK			0x00000700
#define LVT_MT_FIXED		0x00000000
#define LVT_MT_SMI			0x00000200
#define LVT_MT_NMI			0x00000400
#define LVT_MT_EXTINT		0x00000700
#define LVT_DS				0x00001000
#define LVT_RIR				0x00004000
#define LVT_TGM				0x00008000
#define LVT_M				0x00010000
#define LVT_TMM				0x00020000

#define LAPIC_TIMER_VECTOR	(LINT_START_VECTOR + 0)

bool lapic_init( void )
{
	const u64 apic_base = rdmsr( MSR_APIC_BASE );
	if ( apic_base >= 0x100000000LL )
		return false;	// 32ビットアドレスのみ
	if ( !(apic_base & APIC_BASE_BSC) )
		return false;	// APの処理は未実装

	u8 *regs = (u8 *)((u32)apic_base & 0xFFFFF000);

	cpu_disable_interrupt();

	if ( !(apic_base & APIC_BASE_AE) )
		wrmsr( MSR_APIC_BASE, apic_base | APIC_BASE_AE );

	*(u32 *)&regs[LAPIC_LVT_TIMER] = LVT_M | LAPIC_TIMER_VECTOR;
	*(u32 *)&regs[LAPIC_LVT_THERM] = LVT_M;
	*(u32 *)&regs[LAPIC_LVT_PCNTR] = LVT_M;
	*(u32 *)&regs[LAPIC_LVT_ERROR] = LVT_M;
	*(u32 *)&regs[LAPIC_LVT_LINT0] = LVT_MT_EXTINT;
	*(u32 *)&regs[LAPIC_LVT_LINT1] = LVT_MT_NMI;

	cpu_enable_interrupt();

	return true;
}

void lapic_enable_lint0( void )
{
	u8 *regs = (u8 *)((u32)rdmsr( MSR_APIC_BASE ) & 0xFFFFF000);
	*(u32 *)&regs[LAPIC_LVT_LINT0] &= ~LVT_M;
}

void lapic_disable_lint0( void )
{
	u8 *regs = (u8 *)((u32)rdmsr( MSR_APIC_BASE ) & 0xFFFFF000);
	*(u32 *)&regs[LAPIC_LVT_LINT0] |= LVT_M;
}

void lapic_set_timer( u32 cnt, u32 div, bool periodic )
{
	u8 *regs = (u8 *)((u32)rdmsr( MSR_APIC_BASE ) & 0xFFFFF000);
	*(u32 *)&regs[LAPIC_LVT_TIMER] |= LVT_M;
	if ( periodic )
		*(u32 *)&regs[LAPIC_LVT_TIMER] |= LVT_TMM;
	else
		*(u32 *)&regs[LAPIC_LVT_TIMER] &= ~LVT_TMM;
	*(u32 *)&regs[LAPIC_TIMER_ICR] = cnt;
	*(u32 *)&regs[LAPIC_TIMER_DCR] = div;
}

EXTERN_C void CALLBACK local_int_handler( u8 vector )
{
	vmm_printf( VMM_DEBUG, "lapic: Local Interrupt !! (vector=%02x)\n", vector );
}
