/*!
******************************************************************************

	@file	irq.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "printf.h"
#include "cpu.h"
#include "task.h"
#include "atomic.h"
#include "irq.h"
#include "io.h"
#include "pic.h"
#include "disp.h"

static IRQ_INFO irq_info[IRQ_MAX];
static IRQ_INFO irq_info_level[IRQ_MAX][IRQ_LEVEL_ENTRY_MAX];
//static volatile int irq_lock_count;
//static volatile int irq_lock_processing;

static void irq_set_trigger( u8 irq, IRQ_TRIGGER trigger );
static bool irq_level_handler( u8 irq, void *args );

void irq_init( void )
{
	for ( int i=0; i<IRQ_MAX; i++ ) {
		irq_info[i].trigger = IRQ_TRIGGER_EDGE;
		irq_info[i].proc	= NULL;
		irq_info[i].args	= NULL;
	}

	for ( int i=0; i<IRQ_MAX; i++ ) {
		for ( int j=0; j<IRQ_LEVEL_ENTRY_MAX; j++ ) {
			irq_info_level[i][j].trigger = IRQ_TRIGGER_LEVEL;
			irq_info_level[i][j].proc	 = NULL;
			irq_info_level[i][j].args	 = NULL;
		}
	}

//	irq_lock_count = 0;
//	irq_lock_processing = 0;
}

bool irq_register( u8 irq, IRQ_TRIGGER trigger, IRQ_HANDLER proc, void *args )
{
	if ( irq >= IRQ_MAX ) {
		vmm_printf( VMM_DEBUG, "irq(%d): register failed. (irq >= IRQ_MAX)\n",
					irq, trigger, proc, args );
		return false;
	}

	if ( proc == NULL ) {
		vmm_printf( VMM_DEBUG, "irq(%d): register failed. (proc == NULL)\n",
					irq, trigger, proc, args );
		return false;
	}

	if ( trigger == IRQ_TRIGGER_EDGE )
	{	// エッジトリガ
		IRQ_INFO *info = &irq_info[irq];
		if ( info->proc != NULL ) {
			vmm_printf( VMM_DEBUG, "irq(%d): register failed. (trigger=EDGE, proc=%08x, args=%08x)\n",
						irq, proc, args );
			return false;
		}
		info->trigger = IRQ_TRIGGER_EDGE;
		info->proc	  = proc;
		info->args	  = args;
		irq_set_trigger( irq, IRQ_TRIGGER_EDGE );
		pic_enable_irq( irq );
	}
	else
	{	// レベルトリガ
		int i;
		for ( i=0; i<IRQ_LEVEL_ENTRY_MAX; i++ ) {
			IRQ_INFO *info = &irq_info_level[irq][i];
			if ( info->proc == NULL ) {
				info->proc = proc;
				info->args = args;
				break;
			}
		}
		if ( i >= IRQ_LEVEL_ENTRY_MAX ) {
			vmm_printf( VMM_DEBUG, "irq(%d): register failed. (trigger=LEVEL, proc=%08x, args=%08x)\n",
						irq, proc, args );
			return false;
		}
		IRQ_INFO *info = &irq_info[irq];
		if ( info->proc == NULL ) {
			info->trigger = IRQ_TRIGGER_LEVEL;
			info->proc	  = irq_level_handler;
			info->args	  = NULL;
			irq_set_trigger( irq, IRQ_TRIGGER_LEVEL );
			pic_enable_irq( irq );
		}
	}
//	vmm_printf( VMM_DEBUG, "irq(%d): registered. (trigger=%d, proc=%08x, args=%08x)\n",
//				irq, trigger, proc, args );

	return true;
}

void irq_unregister( u8 irq, IRQ_HANDLER proc )
{
	if ( irq >= IRQ_MAX ) {
		vmm_printf( VMM_DEBUG, "irq(%d): unregister failed. (irq >= IRQ_MAX)\n", irq );
		return;
	}

	if ( irq_info[irq].trigger == IRQ_TRIGGER_EDGE )
	{	// エッジトリガ
		IRQ_INFO *info = &irq_info[irq];
		if ( info->proc != proc ) {
			vmm_printf( VMM_DEBUG, "irq(%d): unregister failed. (proc=%08x)\n", irq, proc );
			return;
		}
		info->proc = NULL;
		info->args = NULL;
		pic_disable_irq( irq );
	}
	else
	{	// レベルトリガ
		int i;
		for ( i=0; i<IRQ_LEVEL_ENTRY_MAX; i++ ) {
			IRQ_INFO *info = &irq_info_level[irq][i];
			if ( info->proc == proc ) {
				info->proc = NULL;
				info->args = NULL;
				break;
			}
		}
		if ( i >= IRQ_LEVEL_ENTRY_MAX ) {
			vmm_printf( VMM_DEBUG, "irq(%d): unregister failed. (proc=%08x)\n", irq, proc );
			return;
		}
	}
//	vmm_printf( VMM_DEBUG, "irq(%d): unregistered.\n" );
}

void irq_set_trigger( u8 irq, IRQ_TRIGGER trigger )
{
#ifndef _VSUN86_PCSIM
	irq &= 0x0F;

	u16 port;
	u8  mask;
	if (irq < 8) {
		port = 0x04D0;
		mask = 0xF8;	// bit2-0は常に0
	}
	else {
		port = 0x04D1;
		mask = 0xDE;	// bit5,0は常に0
		irq -= 8;
	}
	u8 elcr = inb( port );
	if ( trigger == IRQ_TRIGGER_LEVEL )
		elcr |= 1 << irq;
	else
		elcr &= ~(1 << irq);
	outb( port, elcr & mask );
#else	//_VSUN86_PCSIM
	(void)irq;
	(void)trigger;
#endif	//_VSUN86_PCSIM
}

EXTERN_C void CALLBACK irq_handler( u8 n )
{
	const u8 irq = (n - IRQ_START_VECTOR) & 0x0F;

	/*
	// 画面右上に割り込み状態を表示する
	const char *p = "0123456789ABCDEF";
	static u16 color[16] = { 0 };
	color[irq] += 0x0100;
	if (color[irq] >= 0x1000)
		color[irq] = 0x0100;
	const bool drawable_bak = disp_get_char_drawable();
	disp_set_char_drawable( true );
	disp_write_char( 64+irq, 0, p[irq], color[irq] );
	disp_set_char_drawable( drawable_bak );
	*/

	// IRQを処理する
	IRQ_INFO *info = &irq_info[irq];
	if ( info->proc != NULL )
		info->proc( irq, info->args );

	// EOIを送る
	pic_send_eoi( irq );

	// (必要に応じて)タスクを切り替える
	task_switch( false, TASK_STATE_BUSY );
}

static bool irq_level_handler( u8 irq, void *args )
{
	(void)args;

	for ( int i=0; i<IRQ_LEVEL_ENTRY_MAX; i++ ) {
		IRQ_INFO *info = &irq_info_level[irq][i];
		if ( info->proc != NULL )
			info->proc( irq, info->args );
	}

	return true;
}

/*
void irq_lock( void )
{
	lock( &irq_lock_processing );

	if ( irq_lock_count == 0x7FFFFFFF ) {
		vmm_printf( VMM_ERROR, "ERROR: irq_lock() failed.\n" );
		abort();
	}
	else if ( irq_lock_count == 0 )
		cpu_disable_interrupt();
	irq_lock_count++;

	unlock( &irq_lock_processing );
}

void irq_unlock( void )
{
	lock( &irq_lock_processing );

	if ( irq_lock_count <= 0 ) {
		vmm_printf( VMM_ERROR, "ERROR: irq_unlock() failed.\n" );
		abort();
	}

	irq_lock_count--;
	if ( irq_lock_count == 0 )
		cpu_enable_interrupt();

	unlock( &irq_lock_processing );
}
*/
