/*!
******************************************************************************

	@file	vdisk.cpp

	Copyright (C) 2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "printf.h"
#include "disk.h"
#include "disk/vdisk.h"

#include <sys/stat.h>
#include <stdio.h>

static const char *vdisk_devname[6] = {
	"fdd0",	"fdd1",	"ata0",	"ata1",	"ata2",	"ata3"
};

static const char *vdisk_filename[6] = {
	"fdd0.img",	"fdd1.img",
	"ata0.img",	"ata1.img",	"ata2.img",	"ata3.img"
};

static FILE *vdisk_fp[6];

bool vdisk_start( void )
{
	vmm_printf( VMM_INFO, "\nInitializing Virtual Disk Devices ...\n\n" );

	for ( int i=0; i<6; i++ ) {
		vdisk_fp[i] = fopen( vdisk_filename[i], "r+b" );
		if ( vdisk_fp[i] == NULL ) {
			vmm_printf( VMM_DEBUG, "fopen(\"%s\") failed.\n", vdisk_filename[i] );
			vmm_printf( VMM_INFO, "%4s: (No Device)\n", vdisk_devname[i] );
			continue;
		}
		u64 capacity;
		if ( !disk_register_vdisk( i, &capacity ) ) {
			vmm_printf( VMM_DEBUG, "disk_register_vdisk(%d) failed.\n", i );
			vmm_printf( VMM_INFO, "%4s: (No Device)\n", vdisk_devname[i] );
			fclose( vdisk_fp[i] );
			vdisk_fp[i] = NULL;
			continue;
		}
		vmm_printf( VMM_INFO, "%4s: %s (%llu bytes)\n", vdisk_devname[i], vdisk_filename[i], capacity );
	}

	return true;
}

bool vdisk_get_capacity( int vdisk_index, u64 *capacity )
{
	struct stat st;

	if ( 0 != fstat( fileno(vdisk_fp[vdisk_index]), &st ) ) {
		vmm_printf( VMM_DEBUG, "fstat() failed.\n" );
		return false;
	}
	*capacity = st.st_size;

	return true;
}

bool vdisk_read( int vdisk_index, u32 block_addr, u16 block_len, void *buf )
{
	FILE *fp = vdisk_fp[vdisk_index];
	if ( block_addr >= 0x400000 ) {
		vmm_printf( VMM_DEBUG, "block_addr >= 0x400000\n" );
		return false;
	}

	if ( 0 != fseek( fp, (long)(block_addr * 512), SEEK_SET ) ) {
		vmm_printf( VMM_DEBUG, "fseek() failed.\n" );
		return false;
	}

	if ( (u32)(block_len * 512) != fread( buf, 1, block_len * 512, fp ) ) {
		vmm_printf( VMM_DEBUG, "fread() failed.\n" );
		return false;
	}

	return true;
}
