/*!
******************************************************************************

	@file	time.c

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include <time.h>

#include "../vmm/vsun86.h"
#include "../vmm/syscall.h"

#define TIME_CONST	0xE7926B380LL

long _timezone = 0;
static struct tm _tm;

// 3/1を基準にした各月の1日までの日数:   3/1  4/1  5/1  6/1  7/1  8/1  9/1 10/1 11/1 12/1  1/1  2/1
static const int days_from_march[12] = {   0,  31,  61,  92, 122, 153, 184, 214, 245, 275, 306, 337 };

// ※紀元1年3月1日は水曜日らしい
static const int wday[7] = { 3, 4, 5, 6, 0, 1, 2 };

struct tm * gmtime( const time_t *timep )
{
	if ( timep == NULL )
		return NULL;

	time_t t = *timep % 86400;
	if ( t < 0 )
		t += 86400;
	const time_t day = (*timep + TIME_CONST - 86400) / 86400;
	const int _4year_days   =   1461;	// 4年   =   1年* 4+1 =   1,461日
	const int _100year_days =  36524;	// 100年 =   4年*25-1 =  36,524日
	const int _400year_days = 146097;	// 400年 = 100年* 4+1 = 146,097日

	const int q_400 = day / _400year_days;
	const int r_400 = day - q_400 * _400year_days;
	const int q_100 = r_400 / _100year_days;
	if ( q_100 == 4 )
	{	// 400年周期の最終日((q_400+1)*400年2月29日)
		_tm.tm_year	= (q_400 + 1) * 400 - 1900;
		_tm.tm_mon	= 1;
		_tm.tm_mday	= 29;
		_tm.tm_yday = 59;
	}
	else {
		const int r_100 = r_400 - q_100 * _100year_days;
		const int q_4 = r_100 / _4year_days;
		const int r_4 = r_100 - q_4 * _4year_days;
		const int q_1 = r_4 / 365;
		_tm.tm_year = (q_400 * 400) + (q_100 * 100) + (q_4 * 4) + q_1 - 1900;
		if ( q_1 == 4 )
		{	// 4年周期の最終日(2/29)
			_tm.tm_mon	= 1;
			_tm.tm_mday	= 29;
			_tm.tm_yday = 59;
		}
		else {
			const int r_1 = r_4 - q_1 * 365;
			if ( r_1 < days_from_march[10] )
			{	// 3/1～12/31
				int i;
				for ( i=1; i<11; i++ ) {
					if ( r_1 < days_from_march[i] ) {
						_tm.tm_mon  = i+1;
						_tm.tm_mday = r_1 - days_from_march[i-1] + 1;
						break;
					}
				}
				_tm.tm_yday = 59 + r_1;
			}
			else
			{	// 翌年1/1～2/28
				_tm.tm_year++;
				if ( r_1 < days_from_march[11] )
				{	// 1/1～1/31
					_tm.tm_mon  = 0;
					_tm.tm_mday = r_1 - days_from_march[10] + 1;
					_tm.tm_yday = _tm.tm_mday - 1;
				}
				else
				{	// 2/1～2/28
					_tm.tm_mon  = 1;
					_tm.tm_mday = r_1 - days_from_march[11] + 1;
					_tm.tm_yday = 31 + _tm.tm_mday - 1;
				}
			}
		}
	}
	_tm.tm_wday = wday[day % 7];
	_tm.tm_hour = t / 3600;
	t -= _tm.tm_hour * 3600;
	_tm.tm_min  = t / 60;
	t -= _tm.tm_min * 60;
	_tm.tm_sec  = t;

	return &_tm;
}

struct tm * localtime( const time_t *timep )
{
	time_t t;

	if ( timep == NULL )
		return NULL;

	t = *timep + _timezone;
	return gmtime( &t );
}

time_t mktime( struct tm *tp )
{
	int year, month;
	time_t t;

	if ( tp == NULL )
		return -1;

	year  = tp->tm_year + 1900;
	month = tp->tm_mon  + 1;
	if ( (year < 0) || (month < 1) || (month > 12) )
		return -1;
	if ( month < 3 )
	{	// 1月、2月は前年の13月、14月として扱う
		if ( year < 1 )
			return -1;
		year--;
		month += 12;
	}

	t = year * 365 + (year / 4) - (year / 100) + (year / 400) + days_from_march[month - 3] + tp->tm_mday;
	t = t * 86400 + tp->tm_hour * 3600 + tp->tm_min * 60 + tp->tm_sec - (TIME_CONST + _timezone);
	return t;
}
