/*!
******************************************************************************

	@file	stdarg.c

	Copyright (C) 2008 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include <stdarg.h>
#include <stdio.h>
#include <ctype.h>

#define FMT_LEFT		0x01
#define FMT_ZEROPAD		0x02
#define FMT_UPPER		0x04
#define FMT_SIGNED		0x08
#define FMT_LONG		0x10
#define FMT_LONGLONG	0x20

int vprintf( const char *fmt, va_list ap )
{
	char buf[256];
	const int n = vsnprintf( buf, sizeof(buf), fmt, ap );
	puts( buf );
	return n;
}

int vsprintf( char *str, const char *fmt, va_list ap )
{
	return vsnprintf( str, 0x7FFFFFFF, fmt, ap );
}

int vsnprintf( char *str, size_t len, const char *fmt, va_list ap )
{
	size_t i = 0, off;
	unsigned char flags;
	int flags_end;
	int column;
	int base;
	int n, mod;
	char num_buf[32];
	int  num_buf_used;
	char *num_fmt;

	for ( off = 0; fmt[off] != '\0'; off++ ) {
		if ( i >= len ) {
			i = len - 1;
			break;
		}
		if ( fmt[off] != '%' ) {
			str[i++] = fmt[off];
			continue;
		}

		off++;
		if ( fmt[off] == '%' ) {	// "%%" -> "%"
			str[i++] = '%';
			continue;
		}

		flags = 0;
		flags_end = 0;
		while ( !flags_end ) {
			switch ( fmt[off] ) {
			case '-':	flags |= FMT_LEFT;		break;	// 左寄せ
			case '0':	flags |= FMT_ZEROPAD;	break;	// 余白をゼロで埋める
			default:	flags_end = 1;			continue;
			}
			off++;
		}

		// 桁数指定
		if ( isdigit( fmt[off] ) ) {	// 桁数指定あり
			column = fmt[off++] - '0';
			while ( isdigit( fmt[off] ) )
				column = column * 10 + fmt[off++] - '0';
			if ( i + column >= len )
				break;
		}
		else if ( fmt[off] == '*' ) {	// 桁数を引数で指定
			off++;
			column = va_arg( ap, int );
			if ( column < 0 ) {			// 負数なら左寄せ
				column = -column;
				flags |= FMT_LEFT;
			}
		}
		else {
			column = 0;
		}

		base = 10;	// デフォルトの基底(=10進数)

		// 長さ修飾子を処理する
		flags_end = 0;
		while ( !flags_end ) {
			switch ( fmt[off] )
			{
			case 'l':
				if ( flags & FMT_LONG )
					flags |= FMT_LONGLONG;
				else
					flags |= FMT_LONG;
				break;
			case 'L':
				flags |= FMT_LONGLONG;
				break;
			default:
				flags_end = 1;
				continue;
			}
			off++;
		}

		// 変換指定子を処理する
		switch ( fmt[off] )
		{
		case 'c':	// 文字
			{
				const int c = va_arg( ap, int );
				if ( i < len )
					str[i++] = (char)c;
			}
			continue;

		case 's':	// 文字列
			{
				char *p = va_arg( ap, char * );
				if ( p == NULL )
					p = "(NULL)";
				for ( n=0; p[n] != '\0'; n++ ) {
					if ( i >= len )
						break;
					str[i++] = p[n];
				}
				if ( flags & FMT_LEFT ) {
					for ( ; n<column; n++ ) {
						if ( i >= len )
							break;
						str[i++] = ' ';
					}
				}
			}
			continue;

		case 'd':	// 10進数(符号あり)
		case 'i':
			flags |= FMT_SIGNED;
			break;

		case 'u':	// 10進数(符号なし)
			break;

		case 'x':	// 16進数(0-9,a-f)
			base = 16;
			break;

		case 'X':	// 16進数(0-9,A-F)
			base = 16;
			flags |= FMT_UPPER;
			break;
		}

		if ( flags & FMT_UPPER )
			num_fmt = "0123456789ABCDEF";
		else
			num_fmt = "0123456789abcdef";

		num_buf_used = 0;
		if ( flags & FMT_SIGNED )
		{	// 符号あり
			if ( flags & FMT_LONGLONG )
			{	// 64ビット
				const long long num = va_arg( ap, long long );
				long long tmp = num;
				do {
					mod = tmp % base;
					tmp /= base;
					num_buf[num_buf_used++] = num_fmt[mod];
				} while ( tmp );
				if ( num < 0 )
					num_buf[num_buf_used++] = '-';
			}
			else
			{	// 32ビット
				const long num = va_arg( ap, long );
				long tmp = num;
				do {
					mod = tmp % base;
					tmp /= base;
					num_buf[num_buf_used++] = num_fmt[mod];
				} while ( tmp );
				if ( num < 0 )
					num_buf[num_buf_used++] = '-';
			}
		}
		else
		{	// 符号なし
			if ( flags & FMT_LONGLONG )
			{	// 64ビット
				const unsigned long long num = va_arg( ap, unsigned long long );
				unsigned long long tmp = num;
				do {
					mod = tmp % base;
					tmp /= base;
					num_buf[num_buf_used++] = num_fmt[mod];
				} while ( tmp );
			}
			else
			{	// 32ビット
				const unsigned long num = va_arg( ap, unsigned long );
				unsigned long tmp = num;
				do {
					mod = tmp % base;
					tmp /= base;
					num_buf[num_buf_used++] = num_fmt[mod];
				} while ( tmp );
			}
		}

		if ( column > 0 )
		{	// 桁数指定あり
			const char padding = (flags & FMT_ZEROPAD)? '0':' ';
			if ( !(flags & FMT_LEFT) ) {	// 右寄せ
				while ( column > num_buf_used ) {
					str[i++] = padding;
					column--;
				}
			}
			while ( num_buf_used > 0 ) {
				str[i++] = num_buf[--num_buf_used];
				column--;
			}
			while ( column > 0 ) {
				str[i++] = padding;
				column--;
			}
		}
		else
		{	// 桁数指定なし
			while ( num_buf_used > 0 )
				str[i++] = num_buf[--num_buf_used];
		}
	}

	str[i] = '\0';
	return i;
}
