/*!
	boot.c
	Kernel and Setting.
	(C) 2013-2014 VOSystems.
*/

/*
	Vane Operating System
	Publish under The MIT License

	MIT License
	http://opensource.org/licenses/mit-license.php
	Copyright (C) 2013-2014 VOSystems. All rights reserved.
*/

#include<multiboot.h>
#include<VaneOS.h>
#include<GDT_IDT.h>
#include<KeyBoard.h>
#include<Interrupt.h>
#include<stdio.h>
#include<scancode.h>
#include<memory.h>
#include<Graphic.h>
#include<Message.h>
#include<timer.h>
#include<System.h>
#include<Env.h>
#include<tools.h>

/*PROTOTYPE*/
void outPortByte( unsigned short address, unsigned char value );
unsigned char inPortByte( unsigned short address );
void initPIC( void );
int setPitCounter( int freq, unsigned char counter, unsigned char mode );
void initPit( void );
void clear_state(void);
void hlt(void);

/*!
	load.S call this function
	
	Set All & Call shell
	(C)2014 VOSystems.
	create 2014/09/23
	update 2014/10/15
	author VOSystems
*/
void cstart(unsigned long magic, unsigned long addr)
{
	/*Common Settings*/
	char cmd[100]={0};
	struct MEMMAN *memman=(struct MEMMAN*)MEMMAN_ADDR;
	multiboot_info_t *mbi;
	
	mbi = (multiboot_info_t *) addr;
		
	vga_text_init();
	break_flag=0;	system_key_code_flag=0;
	screen=(unsigned short*)0xb8000;//VRam Setting
	attr=WHITE;
	
	set_env();
	
	/*Start Set Main Code*/
	printVersion();
	putchar('\n');
	
	/*Setting IDT*/
	printTASK("Setting IDT.....");
	setupInterruptGate(DEF_IDT_INT_NUM_IRQ1,KeyBoard_Interrupt_asm);//Register Keyboard interrupt handler
	setupInterruptGate(DEF_IDT_INT_NUM_IRQ0,Timer_Interrupt_asm);//Register Timer interrupt handler
	idtr.size	= NUM_IDT * sizeof( GATE_DESCRIPTOR );
	idtr.base	= ( GATE_DESCRIPTOR *)idt;
	
	load_idt();
	printOK("Setting IDT.....");
	
	/*Setting GDT*/
	printTASK("Setting GDT.....");
	/*set up null descriptor*/
	gdt[NULL_DESCRIPTOR].limitLo	= DEF_GDT_NULL_LIMIT;
	gdt[NULL_DESCRIPTOR].baseLo		= DEF_GDT_NULL_BASELO;
	gdt[NULL_DESCRIPTOR].baseMid	= DEF_GDT_NULL_BASEMID;
	gdt[NULL_DESCRIPTOR].flags		= DEF_GDT_NULL_FLAGS;
	gdt[NULL_DESCRIPTOR].baseHi		= DEF_GDT_NULL_BASEHI;
	/* set up code descriptor */
	gdt[CODE_DESCRIPTOR].limitLo	= DEF_GDT_CODE_LIMIT;
	gdt[CODE_DESCRIPTOR].baseLo		= DEF_GDT_CODE_BASELO;
	gdt[CODE_DESCRIPTOR].baseMid	= DEF_GDT_CODE_BASEMID;
	gdt[CODE_DESCRIPTOR].flags		= DEF_GDT_CODE_FLAGS;
	gdt[CODE_DESCRIPTOR].baseHi		= DEF_GDT_CODE_BASEHI;

	/* set up data descriptor */
	gdt[DATA_DESCRIPTOR].limitLo	= DEF_GDT_DATA_LIMIT;
	gdt[DATA_DESCRIPTOR].baseLo		= DEF_GDT_DATA_BASELO;
	gdt[DATA_DESCRIPTOR].baseMid	= DEF_GDT_DATA_BASEMID;
	gdt[DATA_DESCRIPTOR].flags		= DEF_GDT_DATA_FLAGS;
	gdt[DATA_DESCRIPTOR].baseHi		= DEF_GDT_DATA_BASEHI;
	
	gdtr.size	= NUM_GDT * sizeof( SEGMENT_DESCRIPTOR );
	gdtr.base	= ( SEGMENT_DESCRIPTOR *)gdt;
	
	load_gdt();
	printOK("Setting GDT.....");
	
	printTASK("Init PIC.....");
	initPIC();
	printOK("Init PIC.....");
	
	printTASK("Init Keyboard.....");
	initKeyboard();
	keyboardEnableKBD();
	printOK("Init Keyboard.....");
	
	printTASK("Init Messaging Buffer.....");
	msg_init(&kernel,32,kernelbuf);
	printOK("Init Messaging Buffer.....");
	
	printTASK("Init PIT.....");
	setPitCounter(100,DEF_PIT_COM_COUNTER0,DEF_PIT_COM_MODE_SQUAREWAVE);
	initPit();
	printOK("Init PIT.....");
	
	printTASK("Starting Memory Manager.....");
	memman_init(memman);
	memman_free(memman,0x00001000,0x0009e000);
	memman_free(memman,0x00400000,((mbi->mem_lower+mbi->mem_upper)*1024)-0x00400000);
	printOK("Starting Memory Manager.....");
	
	printTASK("Loading Memory Info.....");
	if(magic!=MULTIBOOT_BOOTLOADER_MAGIC){
		printFAIL("Loading Memory Info.....");
		
	}else{
		printOK("Loading Memory Info.....");
		printINFO("TotalMemory = ");
		printf("%d MiB\n",(mbi->mem_lower+mbi->mem_upper+1024)/1024);
		printINFO("Free Memory = ");
		printf("%d MiB\n",memman_total(memman)/(1024*1024));
	}
	printINFO("Boot Time and Day = ");
	printtime();
	cr();cr();
	
	do{
		ShowPrompt();//puts("\nKernelShell/Root/>");
		scan_str(cmd,sizeof(cmd)/sizeof(cmd[0]));
		
		if(strcmp("time",cmd)==0){
			int in=0;
			while(in!='q'){
				putchar('\r');
				printtime();
				if(kbhit()=='q')break;
				io_hlt();
				io_hlt();
				io_hlt();
				io_hlt();
				io_hlt();
			}
			cr();
		
		}else if(strcmp("vane",cmd)==0){
			attr=CH_BLUE;
			printVersion();
			attr=WHITE;
			cr();cr();
			
		}else if(strcmp("copyright",cmd)==0){
			attr=CH_GREE;
			printCopyRight();
			attr=WHITE;
			cr();
			
		}else if(strcmp("cls",cmd)==0 || strcmp("clear",cmd)==0){
			ClearScreen();
			
		}else{
			if(strcmp("exit",cmd)==0)break;
			printFAIL("Unknown this command");
			cr();
			
		}
		//init_char_arrey(cmd,100);
	}while(strcmp(cmd,"exit")!=0);
	while(1)io_hlt();
	return;
}

void outPortByte( unsigned short address, unsigned char value )
{
	__asm__ __volatile__( "out %%dx, %%al" : : "d"(address), "a"(value) );
}

unsigned char inPortByte( unsigned short address )
{
	unsigned char data;

	__asm__ __volatile__( "mov %%dx, %%ax" : : "a"(address) );
	__asm__ __volatile__( "in %al, %dx" );
	__asm__ __volatile__( "mov %%bl, %%al" : "=b"(data) );
	return( data );
}

void initPIC( void )
{
	disable();
	
	outPortByte( PORT_MASTER_PIC_COMMAND,	PIC_ICW1	);
	outPortByte( PORT_SLAVE_PIC_COMMAND,	PIC_ICW1	);

	outPortByte( PORT_MASTER_PIC_DATA,	PIC_MASTER_ICW2	);
	outPortByte( PORT_SLAVE_PIC_DATA,	PIC_SLAVE_ICW2	);

	outPortByte( PORT_MASTER_PIC_DATA,	PIC_MASTER_ICW3	);
	outPortByte( PORT_SLAVE_PIC_DATA,	PIC_SLAVE_ICW3	);

	outPortByte( PORT_MASTER_PIC_DATA,	PIC_MASTER_ICW4	);
	outPortByte( PORT_SLAVE_PIC_DATA,	PIC_SLAVE_ICW4	);
	
	outPortByte( PORT_MASTER_PIC_IMR, (~PIC_IMR_MASK_IRQ0) & (~PIC_IMR_MASK_IRQ2 )	);
	outPortByte( PORT_SLAVE_PIC_IMR, PIC_IMR_MASK_IRQ_ALL				);
	
    outPortByte( PORT_MASTER_PIC_IMR, ( ~PIC_IMR_MASK_IRQ0 )
                 & ( ~PIC_IMR_MASK_IRQ1 ) & ( ~PIC_IMR_MASK_IRQ2 ) );
    outPortByte( PORT_SLAVE_PIC_IMR, PIC_IMR_MASK_IRQ_ALL          );
    
    enable();
}

int setPitCounter( int freq, unsigned char counter, unsigned char mode )
{
	unsigned short	count;
	unsigned char	command;

	/* --------------------------------------------------------------------- */
	/* 	calculate frequency						 */
	/* --------------------------------------------------------------------- */
	count = ( unsigned short )( DEF_PIT_CLOCK / freq );

	/* --------------------------------------------------------------------- */
	/* 	make initial command						 */
	/* --------------------------------------------------------------------- */
	command = mode | DEF_PIT_COM_RL_DATA | counter;

	outPortByte( PIT_REG_CONTROL, command );

	/* --------------------------------------------------------------------- */
	/* 	send counter value						 */
	/* --------------------------------------------------------------------- */
	outPortByte( PIT_REG_COUNTER0, ( unsigned char )( count & 0xFF		) );
	outPortByte( PIT_REG_COUNTER0, ( unsigned char )( ( count >> 8 ) & 0xFF	) );
}

/*
	Funtion     :initPit
	Input       :void
	Output      :void
	Return      :void 

	Description :initialize pic
*/
void initPit( void )
{
	/* --------------------------------------------------------------------- */
	/* 	set counter0 100Hz						 */
	/* --------------------------------------------------------------------- */
	setPitCounter( 100, DEF_PIT_COM_COUNTER0,  DEF_PIT_COM_MODE_SQUAREWAVE );
	timer.count=0;
	int i;
	for(i=0;i<MAX_TIMER;i++)
	{
		timer.timers[i]->flags=0;
	}
}
