#!/usr/bin/env python
# -*- coding: utf-8 -*-
#   Copyright 2008 Agile42 GmbH - Andrea Tomasini 
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
# 
# Authors:
#     - Andrea Tomasini <andrea.tomasini_at_agile42.com>

import unittest

from agilo.test import BetterEnvironmentStub
from agilo.init import AgiloInit
from agilo.utils.config import AgiloConfig


class InstallationTest(unittest.TestCase):
    """Tests agilo installation process, calling the main init method"""
    
    def setUp(self):
        self.env = self.initialize_and_upgrade_agilo()
        self.config = AgiloConfig(self.env)
    
    def initialize_and_upgrade_agilo(self, enabled_third_party_components=()):
        enabled_components = ['trac.*', 'agilo.*'] + list(enabled_third_party_components)
        env = BetterEnvironmentStub(default_data=True, enable=enabled_components)
        for component_key in enabled_components:
            env.config.set('components', component_key, 'enabled')
        
        ai = AgiloInit(env)
        db = env.get_db_cnx()
        if ai.environment_needs_upgrade(db):
            ai.upgrade_environment(db)
        return env
    
    def testAgiloPolicyIsAdded(self):
        self.assertEquals(self.config.get('permission_policies', section='trac'),
                          u'AgiloPolicy, DefaultPermissionPolicy, LegacyAttachmentPolicy')
    
    def testTemplatesDirIsSet(self):
        template_path = self.config.get('templates_dir', section='inherit')
        self.assertTrue(template_path.find('templates') > -1)
    
    def is_email_verification_enabled(self, env):
        try:
            from acct_mgr.web_ui import EmailVerificationModule
        except ImportError:
            return None
        return env.is_component_enabled(EmailVerificationModule)
    
    def assert_status_of_email_verification_module(self, env, expected):
        is_enabled = self.is_email_verification_enabled(env)
        if is_enabled is None:
            return
        self.assertEqual(expected, is_enabled)
    
    def testEmailVerificationModuleIsDisabled(self):
        env = self.initialize_and_upgrade_agilo(['acct_mgr.*'])
        self.assert_status_of_email_verification_module(env, False)
    
    def testEmailVerificationStaysEnabledIfEnabledExplicitelyBefore(self):
        env = self.initialize_and_upgrade_agilo(['acct_mgr.*', 'acct_mgr.web_ui.emailverificationmodule'])
        self.assert_status_of_email_verification_module(env, True)


if __name__ == '__main__':
    suite = unittest.TestLoader().loadTestsFromTestCase(InstallationTest)
    unittest.TextTestRunner(verbosity=0).run(suite)