# -*- coding: utf-8 -*-
#   Copyright 2009 Agile42 GmbH
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
# 
# Authors:
#     - Felix Schwarz <felix.schwarz__at__agile42.com>

from unittest import TestCase

from trac.util.compat import set

from agilo.scrum.backlog.json_ui import ConfiguredChildTypesView
from agilo.test import TestEnvHelper
from agilo.utils import Action, Key, Role, Type

class ChildBreakDownTest(TestCase):
    
    def setUp(self):
        self.teh = TestEnvHelper()
        self.env = self.teh.get_env()
        self.view = ConfiguredChildTypesView(self.env)
    
    def call_view(self):
        req = self.teh.mock_request()
        return self.view.do_get(req, {})
    
    def grant_permission(self, action):
        self.teh.grant_permission('anonymous', action)
    
    def test_list_possible_types(self):
        default_types = set([Type.REQUIREMENT, Type.USER_STORY, Type.TASK, Type.BUG])
        self.assertEqual(default_types, set(self.call_view().keys()))
    
    def test_show_possible_children(self):
        self.grant_permission(Action.TRAC_ADMIN)
        link_tree = self.call_view()
        self.assertEqual([Type.USER_STORY], link_tree[Type.REQUIREMENT].keys())
        self.assertEqual([Type.TASK], link_tree[Type.USER_STORY].keys())
        self.assertEqual([], link_tree[Type.TASK].keys())
        self.assertEqual([Type.USER_STORY, Type.TASK], link_tree[Type.BUG].keys())
    
    def test_hide_types_which_the_user_may_not_create(self):
        self.grant_permission(Role.TEAM_MEMBER)
        link_tree = self.call_view()
        self.assertEqual([], link_tree[Type.REQUIREMENT].keys())
    
    def test_show_attributes_which_should_be_copied_for_referenced_tickets(self):
        self.grant_permission(Role.SCRUM_MASTER)
        link_tree = self.call_view()
        self.assertEqual([Type.TASK], link_tree[Type.USER_STORY].keys())
        attributes_to_copy = link_tree[Type.USER_STORY][Type.TASK]
        self.assertEqual(set([Key.SPRINT, Key.OWNER]), set(attributes_to_copy))


