# -*- encoding: utf-8 -*-
#   Copyright 2008 Agile42 GmbH, Berlin (Germany)
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
#   
#   Author: 
#       - Andrea Tomasini <andrea.tomasini__at__agile42.com>

import unittest

from agilo.api.controller import ICommand
from agilo.test import TestEnvHelper
from agilo.scrum.backlog.controller import BacklogController


class BacklogControllerTest(unittest.TestCase):
    """Tests the Backlog Controller"""
    
    def setUp(self):
        """Loads a Backlog Controller"""
        self.teh = TestEnvHelper()
        self.env = self.teh.get_env()
        self.controller = BacklogController(self.env)
        
    def testGetBacklogCommand(self):
        """Tests the GetBacklogCommand"""
        # test with a non existing backlog
        get_backlog = BacklogController.GetBacklogCommand(self.env,
                                                          name='NotExisting')
        not_there = self.controller.process_command(get_backlog)
        self.assertEqual(None, not_there)
        # now create a backlog and test it
        backlog = self.teh.create_backlog('RealOne', num_of_items=10)
        self.assertTrue(backlog.exists)
        self.assertEqual(10, backlog.count())
        
        get_backlog = BacklogController.GetBacklogCommand(self.env,
                                                          name='RealOne')
        real_one = self.controller.process_command(get_backlog)
        self.assertNotEqual(None, real_one)
        self.assertTrue(real_one.exists)
        self.assertEqual(10, real_one.count())
        # now try with a Sprint Backlog
        sprint = self.teh.create_sprint('MySprint')
        backlog = self.teh.create_backlog('MySprintBacklog', 
                                          num_of_items=10, 
                                          b_type=1, 
                                          scope=sprint.name)
        self.assertTrue(backlog.exists)
        items_count = backlog.count()
        
        get_backlog = BacklogController.GetBacklogCommand(self.env,
                                                          name='MySprintBacklog',
                                                          scope='MySprint')
        sprint_backlog = self.controller.process_command(get_backlog)
        self.assertNotEqual(None, sprint_backlog)
        self.assertTrue(sprint_backlog.exists)
        self.assertEqual(items_count, sprint_backlog.count())
        
    def testCreateBacklogCommand(self):
        """Test the creation of a Backlog with various parameters"""
        create_cmd = BacklogController.CreateBacklogCommand(self.env,
                                                            name='TestBacklog')
        backlog = self.controller.process_command(create_cmd)
        self.assertNotEqual(None, backlog)
        self.assertTrue(backlog.exists)
        
        # Try to create a backlog that already exist
        backlog = self.controller.process_command(create_cmd)
        self.assertEqual(None, backlog)
        
    def testMoveBacklogItemCommand(self):
        """Test the moving of a backlog item in the backlog"""
        backlog = self.teh.create_backlog('MovingBacklog', 
                                          num_of_items=20)
        first_item = backlog[0]
        self.assertEqual(0, first_item.pos)
        # create a moving command and move the first item to the 5th
        # poistion in the backlog
        fifth_item = backlog[4]
        self.assertEqual(4, fifth_item.pos)
        move_cmd = BacklogController.MoveBacklogItemCommand(self.env,
                                                            name='MovingBacklog',
                                                            ticket=first_item,
                                                            to_pos=4)
        self.controller.process_command(move_cmd)
        # we need to reload the backlog from the DB
        get_reload_cmd = BacklogController.GetBacklogCommand(self.env,
                                                             name='MovingBacklog')
        backlog = self.controller.process_command(get_reload_cmd)
        self.assertEqual(backlog[4].ticket, first_item.ticket)
        self.assertEqual(backlog[3].ticket, fifth_item.ticket)
        self.assertEqual(4, backlog[4].pos)
        self.assertEqual(3, backlog[3].pos)
    
    def testSaveBacklogCommand(self):
        """Tests the saving backlog command"""
        backlog = self.teh.create_backlog('ChangingBacklog', num_of_items=20)
        self.assertEqual(20, backlog.count())
        # now remove some of the items and save
        backlog.remove(0, pos=True)
        backlog.remove(1, pos=True)
        self.assertEqual(18, backlog.count())
        # now save it and reload it
        cmd_save_backlog = BacklogController.SaveBacklogCommand(self.env,
                                                                name='ChangingBacklog',
                                                                tickets=backlog._tickets)
        self.assertTrue(self.controller.process_command(cmd_save_backlog))
        # now reload the backlog and check the two items have been removed
        cmd_get_backlog = BacklogController.GetBacklogCommand(self.env,
                                                              name='ChangingBacklog')
        backlog = self.controller.process_command(cmd_get_backlog)
        self.assertEqual(18, backlog.count())
        