﻿using NUnit.Framework;
using Meyn.TestLink;
using CookComputing.XmlRpc;

namespace BTS.Trac.Test
{
    using Severity = TracNameAndValueField<ISeverity>;

    [TestFixture]
    [TestLinkFixture(
        Url = TestLinkConnect.Url,
        ProjectName = TestLinkConnect.ProjectName,
        UserId = TestLinkConnect.UserId,
        TestPlan = TestLinkConnect.TestPlan,
        TestSuite = TestLinkConnect.TestSuite,
        DevKey = TestLinkConnect.DevKey )]
    public class SeverityTest
    {
        [SetUp]
        public void SetUp()
        {
            Trac.Connect( TracConnect.Url, TracConnect.UserName, TracConnect.Password );

            // 最初にすべて削除しておく
            string[] severityNames = Severity.GetAll();
            foreach ( string severityName in severityNames ) {
                Severity.Delete( severityName );
            }
        }

        [TearDown]
        public void TearDown()
        {
            Trac.Close();
        }

        [Test]
        public void 重要度の項目を取得する()
        {
            int count = 10;

            // 作成
            for ( int i = 0; i < count; ++i ) {
                string severityName = "Test" + i.ToString();
                Severity severity = new Severity();
                severity.Value = i.ToString();
                severity.Create( severityName );
            }

            string[] severityNames = Severity.GetAll();
            Assert.AreEqual( severityNames.Length, count );
        }

        [Test]
        public void 重要度の各項目を取得する()
        {
            int count = 10;
            string severityNameBase = "Test";

            // 作成
            for ( int i = 0; i < count; ++i ) {
                string severityName = severityNameBase + i.ToString();
                Severity severity = new Severity();
                severity.Value = i.ToString();
                severity.Create( severityName );
            }

            // 作成数を確認
            string[] severityNames = Severity.GetAll();
            Assert.AreEqual( severityNames.Length, count );

            for ( int i = 0; i < count; ++i ) {
                string severityName = severityNameBase + i.ToString();
                Severity severity = new Severity( severityName );

                // 作成したものと取得したものが一致する
                Assert.AreEqual( severity.Name, severityName );
                Assert.AreEqual( severity.Value, i.ToString() );
            }
        }

        [Test]
        public void 重要度を作成する()
        {
            string[] before = Severity.GetAll();

            // 作成
            string severityName = "Test";
            Severity severity = new Severity();
            severity.Value = "10";
            severity.Create( severityName );

            // 重要度の数が増えていること
            Assert.AreEqual( before.Length + 1, Severity.GetAll().Length );

            // 取得した重要度と一致すること
            Severity getSeverity = new Severity( severityName );
            Assert.AreEqual( severity.Name, getSeverity.Name );
            Assert.AreEqual( severity.Value, getSeverity.Value );
        }

        [Test]
        public void 重要度を更新する()
        {
            string[] before = Severity.GetAll();

            // 作成
            string severityName = "Test";
            Severity severity = new Severity();
            severity.Value = "10";
            severity.Create( severityName );

            // 重要度の数が増えていること
            Assert.AreEqual( before.Length + 1, Severity.GetAll().Length );

            // 更新
            Severity getSeverity = new Severity( severityName );
            getSeverity.Value = "11";
            getSeverity.Update();

            // 更新した重要度と一致すること
            Severity updateSeverity = new Severity( severityName );
            Assert.AreEqual( getSeverity.Name, updateSeverity.Name );
            Assert.AreEqual( getSeverity.Value, updateSeverity.Value );
        }

        [Test]
        public void 重要度を削除する()
        {
            string[] before = Severity.GetAll();

            // 作成
            string severityName = "Test";
            Severity severity = new Severity();
            severity.Value = "10";
            severity.Create( severityName );

            // 重要度の数が増えていること
            Assert.AreEqual( before.Length + 1, Severity.GetAll().Length );

            severity.Delete();

            // 重要度の数が戻っていること
            Assert.AreEqual( before.Length, Severity.GetAll().Length );
        }

        [Test]
        public void 重要度を削除するstatic版()
        {
            string[] before = Severity.GetAll();

            // 作成
            string severityName = "Test";
            Severity severity = new Severity();
            severity.Value = "10";
            severity.Create( severityName );

            // 重要度の数が増えていること
            Assert.AreEqual( before.Length + 1, Severity.GetAll().Length );

            Severity.Delete( severityName );

            // 重要度の数が戻っていること
            Assert.AreEqual( before.Length, Severity.GetAll().Length );
        }

        [Test]
        [Ignore]
        public void 同じ順序の重要度を作成すると例外が発生する()
        {
            ///
            /// 今のところ例外が発生せず正常に登録されてしまう
            ///

            // 作成
            string severityName = "Test";
            Severity severity = new Severity();
            severity.Value = "10";
            severity.Create( severityName );

            severity.Create( severityName + "2" );

            // 同じ順序の重要度を作成すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 同じ名称の重要度を作成すると例外が発生する()
        {
            // 作成
            string severityName = "Test";
            Severity severity = new Severity();
            severity.Value = "10";
            severity.Create( severityName );

            severity.Value = "11";
            severity.Create( severityName );

            // 同じ名称の重要度を作成すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称の重要度を取得すると例外が発生する()
        {
            Severity severity = new Severity( "存在しない重要度" );

            // 登録されていない名称の重要度を取得すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称の重要度を更新すると例外が発生する()
        {
            Severity severity = new Severity();
            severity.Update();

            // 登録されていない名称の重要度を更新すると例外が発生する
            Assert.Fail();
        }

        [Test]
        [ExpectedException( typeof( XmlRpcFaultException ) )]
        public void 登録されていない名称の重要度を削除すると例外が発生する()
        {
            Severity.Delete( "存在しない重要度" );

            // 登録されていない名称の重要度を削除すると例外が発生する
            Assert.Fail();
        }
    }
}
