﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using System.Windows.Forms;
using BTS.Trac;
using CookComputing.XmlRpc;

namespace BTS.Trac.Forms
{
    /// <summary>
    /// カスタムフィールド
    /// </summary>
    public class TicketCustomField
    {
        /// <summary>
        /// 属性
        /// </summary>
        private XmlRpcAttributes Attributes = null;

        /// <summary>
        /// フィールド名称
        /// </summary>
        public string Name
        {
            get
            {
                return Attributes.Get<string>( "name" );
            }
        }

        /// <summary>
        /// タイトル
        /// </summary>
        public Label Caption
        {
            get;
            private set;
        }

        /// <summary>
        /// 値のコントロール
        /// </summary>
        public Control ValueControl
        {
            get;
            private set;
        }

        /// <summary>
        /// 値
        /// </summary>
        public string Value
        {
            get
            {
                // チェックボックスはチェック状態を返す
                if ( ValueControl is CheckBox ) {
                    CheckBox check = ValueControl as CheckBox;
                    return check.Checked ? "1" : "0";
                }
                // ラジオボタンはパネル内のラジオボタンをチェックする
                else if ( ValueControl is Panel ) {
                    Panel panel = ValueControl as Panel;
                    foreach ( RadioButton radio in panel.Controls ) {
                        if ( radio.Checked ) {
                            return radio.Text;
                        }
                    }

                    return "";
                }
                // それ以外はテキストをそのまま返す
                else {
                    return ValueControl.Text;
                }
            }

            set
            {
                // チェックボックスはチェック状態変更する
                if ( ValueControl is CheckBox ) {
                    CheckBox check = ValueControl as CheckBox;
                    check.Checked = value == "1";
                }
                // ラジオボタンはパネル内のラジオボタンをチェックする
                else if ( ValueControl is Panel ) {
                    Panel panel = ValueControl as Panel;
                    foreach ( RadioButton radio in panel.Controls ) {
                        radio.Checked = radio.Text == value;
                    }
                }
                // それ以外はテキストをそのまま設定する
                else {
                    ValueControl.Text = value;
                }
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="Attributes"></param>
        public TicketCustomField( XmlRpcAttributes Attributes )
        {
            this.Attributes = Attributes;
            Caption = new Label()
            {
                AutoSize = false,
                Text = Attributes.Get<string>( "label" ) + "：",
                TextAlign = ContentAlignment.MiddleRight,
                Size = new Size( 100, 12 ),
            };

            ValueControl = CreateControl();
            ValueControl.Size = new Size( 161, 19 );

        }

        /// <summary>
        /// コントロールの作成
        /// </summary>
        /// <param name="control"></param>
        /// <returns></returns>
        private Control CreateControl()
        {
            // 開始予定日と終了日は日付入力
            if ( (Attributes.Get<string>( "name" ) == "due_assign")
              || (Attributes.Get<string>( "name" ) == "due_close") ) {
                return new DateTimePicker()
                {
                    Format = DateTimePickerFormat.Short,
                };
            }
            // 合計作業時間はラベル
            else if ( (Attributes.Get<string>( "name" ) == "totalhours") ) {
                return new Label()
                {
                    Text = Attributes.Get<string>( "value" ),
                };
            }

            switch ( Attributes.Get<string>( "type" ) ) {
            // テキスト
            case "text":
                return new TextBox()
                {
                    Text = Attributes.Get<string>( "value" ),
                };
            // チェックボックス
            case "checkbox":
                return new CheckBox()
                {
                    Checked = int.Parse( Attributes.Get<string>( "value" ) ) == 1,
                };
            // ラジオボタン
            case "radio":
                Panel panel = new Panel();
                Point location = new Point();

                // 選択肢を設定
                string[] radioOptions = Attributes.Get<string[]>( "options" );
                foreach ( string option in radioOptions ) {
                    RadioButton radio = new RadioButton()
                    {
                        AutoSize = true,
                        Text = option,
                        Location = location,
                        // チェック状態を設定
                        Checked = (option == Attributes.Get<string>( "value" )),
                    };

                    panel.Controls.Add( radio );

                    // 表示位置をずらす
                    location.X += radio.Size.Width;
                }

                return panel;
            // コンボボックス
            case "select":
                ComboBox combo = new ComboBox()
                {
                    DropDownStyle = ComboBoxStyle.DropDownList,
                };

                // 選択肢を設定
                string[] options = Attributes.Get<string[]>( "options" );
                if ( options != null ) {
                    combo.Items.Add( "" );
                    foreach ( string option in options ) {
                        combo.Items.Add( option );
                    }

                    // デフォルト値を設定（デフォルト値がなければ先頭を選択させえる）
                    string value = Attributes.Get<string>( "value" );
                    if ( string.IsNullOrEmpty( value ) ) {
                        combo.SelectedIndex = 0;
                    }
                    else {
                        combo.Text = value;
                    }
                }

                return combo;
            // テキストエリア
            case "textarea":
                return new TextBox()
                {
                    AcceptsReturn = true,
                    Multiline = true,
                    ScrollBars = ScrollBars.Both,
                };
            }

            throw new ArgumentException();
        }
    }
}
